import * as React from "react";

import { action, observable } from "mobx";
import { observer } from "mobx-react";

import { Background, Layout, Overflow, Position, SearchInput, StyledLayout, ZIndex } from "twitch-core-ui";

export interface Suggestion {
  value: string | null;
  display?: JSX.Element; // If specified, this will be used to display the option instead of its value.
  match?: (input: string) => boolean; // If specified, this function should return true IFF input matches the suggestion
}

export enum SuggestionDisplay {
  ALWAYS = "always", // Always show suggestions (default behaviour)
  ON_FOCUS = "on_focus", // Show suggestions when the input box is focused
  ON_INPUT = "on_input" // Show suggestions when user types input
}

export interface Props {
  show?: boolean; // If false, the component will not be rendered.
  display?: SuggestionDisplay; // When to display the suggestions
  suggestionCardClassName?: string; // Customized class name to control the look of suggestion list
  strictMatch?: boolean; // If specified, onSelect will only be invoked if input matches a single suggestion
  suggestions: Suggestion[]; // List of suggestions for the given
  onSelect: (value: string | null) => void; // Callback that will be invoked if a suggestion is selected
  floatSuggestions?: boolean; // Determines if the suggestions should "float" above (position: absolute)
  defaultValue?: string; // default value, for IDs resolved previously (i.e. query params)
}

export enum TestSelectors {
  SEARCH_INPUT = "auto-suggest-input--search-input",
  SUGGESTIONS = "auto-suggest-input--suggestions",
  SUGGESTION_ITEM = "auto-suggest-input--suggestion-item"
}

@observer
export class AutoCompleteInput extends React.Component<Props> {
  @observable private input = "";
  @observable private isFocused = false;
  @observable private selectionMade = false;
  @observable private matchedSuggestions: Suggestion[] = [];

  public componentDidMount() {
    this.updateSuggestions();

    if (this.props.defaultValue && this.input === "") {
      this.input = this.props.defaultValue;
      this.selectionMade = true;
    }
  }

  public componentWillReceiveProps(previousProps: Props) {
    // Clear input on hide
    if (this.props.show !== previousProps.show && this.props.show === false) {
      this.input = "";
    }

    // Update available suggestions if display changes
    if (this.props.display !== previousProps.display) {
      this.updateSuggestions();
    }
  }

  public render() {
    const { show, suggestionCardClassName } = this.props;
    if (show === false) {
      return null;
    }

    const suggestions = this.matchedSuggestions.map((s, index) => {
      const onMouseDown = (e: React.MouseEvent<HTMLElement>) => {
        e.stopPropagation();
        this.props.onSelect(s.value);
        if (s.value) {
          this.input = s.value;
          this.selectionMade = true;
        }
      };

      return (
        // onMouseDown triggers before onBlur. This is needed to make only show items on focus
        // to work correctly (order of event triggering is mouseDown -> blur -> mouseUp -> click)
        <Layout fullWidth key={s.value ? s.value : index}>
          <button
            data-track-click={`auto-suggest-select-${s.value}`}
            className="tw-interactable tw-full-width"
            onMouseDown={onMouseDown}
            data-test-selector={TestSelectors.SUGGESTION_ITEM}
          >
            {s.display ? s.display : s.value}
          </button>
        </Layout>
      );
    });

    return (
      <Layout position={Position.Relative}>
        <SearchInput
          onChange={this.onChange}
          onKeyDown={this.onKeyDown}
          onFocus={this.onFocus}
          onBlur={this.onBlur}
          value={this.input}
          data-test-selector={TestSelectors.SEARCH_INPUT}
        />
        {!this.selectionMade &&
          this.matchedSuggestions.length > 0 && (
            <StyledLayout
              border
              className={suggestionCardClassName}
              overflow={Overflow.Scroll}
              background={Background.Alt}
              padding={{ y: 0.5 }}
              data-test-selector={TestSelectors.SUGGESTIONS}
              position={this.props.floatSuggestions ? Position.Absolute : Position.Relative}
              zIndex={ZIndex.Above}
              fullWidth
            >
              {suggestions}
            </StyledLayout>
          )}
      </Layout>
    );
  }

  @action
  private onChange = (e: React.FormEvent<HTMLInputElement>) => {
    this.input = e.currentTarget.value;
    if (e.currentTarget.value === "") {
      this.selectionMade = false;
      this.props.onSelect(null);
    }

    this.updateSuggestions();
  };

  @action
  private onKeyDown = (e: React.KeyboardEvent<HTMLElement>) => {
    const { strictMatch, onSelect } = this.props;
    if (e.key === "Enter" && this.input !== "") {
      this.selectionMade = true;
      if (strictMatch) {
        if (this.matchedSuggestions.length === 1) {
          onSelect(this.matchedSuggestions[0].value);
        }
      } else {
        onSelect(this.input);
      }
    } else {
      this.selectionMade = false;
    }
  };

  @action
  private onFocus = () => {
    this.isFocused = true;
    this.updateSuggestions();
  };

  @action
  private onBlur = () => {
    this.isFocused = false;
    this.updateSuggestions();
  };

  @action
  private updateSuggestions = () => {
    const { display, suggestions } = this.props;

    let availableSuggestions: Suggestion[] = [];
    switch (display) {
      case SuggestionDisplay.ON_FOCUS:
        availableSuggestions = this.input || this.isFocused ? suggestions : [];
        break;
      case SuggestionDisplay.ON_INPUT:
        availableSuggestions = this.input ? suggestions : [];
        break;
      default:
        availableSuggestions = suggestions;
    }

    this.matchedSuggestions = availableSuggestions.filter(s => {
      if (s.match) {
        return s.match(this.input);
      }
      return s.value && s.value.indexOf(this.input) > -1;
    });
  };
}
