import * as React from "react";

import { action, observable } from "mobx";
import { observer } from "mobx-react";

import { AlignItems, BorderRadius, Display, Icon, Layout, StyledLayout, SVGAsset } from "twitch-core-ui";
import { CollapsibleComponent } from "./component";

import "./styles.scss";

export interface Props {
  title: JSX.Element;
  defaultClosed?: boolean;
  roundedCorner?: boolean;
  arrowIconType?: ArrowIconType;
}

export enum ArrowIconType {
  Glyph,
  Angle
}

export enum TestSelector {
  Header = "header"
}

@observer
export class CollapsibleCard extends React.Component<Props> {
  @observable isOpen = !this.props.defaultClosed;

  public render() {
    const { arrowIconType, roundedCorner, title } = this.props;
    let arrow: SVGAsset;
    if (this.isOpen) {
      switch (arrowIconType) {
        case ArrowIconType.Angle:
          arrow = SVGAsset.AngleDown;
          break;
        case ArrowIconType.Glyph:
        // Fallthrough
        default:
          arrow = SVGAsset.GlyphArrDown;
      }
    } else {
      switch (arrowIconType) {
        case ArrowIconType.Angle:
          arrow = SVGAsset.AngleRight;
          break;
        case ArrowIconType.Glyph:
        // Fallthrough
        default:
          arrow = SVGAsset.GlyphArrRight;
      }
    }

    const titleComponent = (
      <div onClick={this.toggleCard} data-test-selector={TestSelector.Header}>
        <StyledLayout
          display={Display.Flex}
          alignItems={AlignItems.Center}
          padding={{ top: 0.5, bottom: 0.5, left: 1, right: 1 }}
          borderRadius={
            roundedCorner
              ? {
                  topLeft: BorderRadius.Medium,
                  topRight: BorderRadius.Medium
                }
              : undefined
          }
          fullWidth
          className="collapsible-card__title"
        >
          <Layout display={Display.Flex} alignItems={AlignItems.Center} padding={{ right: 1 }}>
            <Icon asset={arrow} />
          </Layout>
          {title}
        </StyledLayout>
      </div>
    );

    return (
      <CollapsibleComponent title={titleComponent} open={this.isOpen}>
        {this.props.children}
      </CollapsibleComponent>
    );
  }

  @action
  private toggleCard = () => {
    this.isOpen = !this.isOpen;
  };
}
