import * as React from "react";

import { inject, observer } from "mobx-react";

import { ModalCloser } from "aegis/functionality/components/modal-closer";
import { ModalProps } from "aegis/stores";
import {
  AlignContent,
  AlignItems,
  Background,
  Button,
  ButtonType,
  CoreText,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
  StyledLayout,
  TextType
} from "twitch-core-ui";

import "./styles.scss";

export const ConfirmationModalConfirmSelector = "confirmation-modal.confirm";
export const ConfirmationModalCancelSelector = "confirmation-modal.cancel";

export interface ConfirmationModalProps {
  title: string;
  body: string;
  onCancel?: () => void;
  onConfirm: () => void;
  confirmLabelOverride?: string;
  cancelLabelOverride?: string;
}

type Props = ConfirmationModalProps & ModalProps;

@inject("modal")
@observer
export class ConfirmationModalComponent extends React.Component<Props, {}> {
  cancel = (e?: React.MouseEvent) => {
    if (e) {
      e.stopPropagation();
    }
    if (this.props.onCancel) {
      this.props.onCancel();
    } else {
      this.props.modal!.closeModal();
    }
  };

  confirm = (e?: React.MouseEvent) => {
    if (e) {
      e.stopPropagation();
    }
    this.props.onConfirm();
  };

  render() {
    const { title, body, confirmLabelOverride, cancelLabelOverride } = this.props;
    const confirmLabel = confirmLabelOverride || "Confirm";
    const cancelLabel = cancelLabelOverride || "Cancel";
    return (
      <Layout className="confirmation-modal">
        <StyledLayout
          background={Background.AccentAlt}
          padding={{ x: 0.5, y: 0.5 }}
          alignContent={AlignContent.Center}
          alignItems={AlignItems.Center}
          fullHeight
          fullWidth
        >
          <ModalCloser closeModal={this.cancel} />
          <StyledLayout
            background={Background.Base}
            fullHeight={true}
            alignContent={AlignContent.Center}
            display={Display.Flex}
            flexDirection={FlexDirection.Column}
          >
            <Layout padding={{ y: 1 }}>
              <CoreText type={TextType.H5}>{title}</CoreText>
            </Layout>
            <StyledLayout borderTop borderBottom padding={{ y: 2 }}>
              <CoreText>{body}</CoreText>
            </StyledLayout>
            <Layout display={Display.Flex} justifyContent={JustifyContent.Center} padding={{ y: 2 }}>
              <Layout padding={{ right: 1 }}>
                <Button
                  data-track-click="confirmation-modal-cancel"
                  type={ButtonType.Hollow}
                  onClick={this.cancel}
                  data-test-selector={ConfirmationModalCancelSelector}
                >
                  {cancelLabel}
                </Button>
              </Layout>
              <Button
                data-track-click="confirmation-modal-submit"
                onClick={this.confirm}
                data-test-selector={ConfirmationModalConfirmSelector}
              >
                {confirmLabel}
              </Button>
            </Layout>
          </StyledLayout>
        </StyledLayout>
      </Layout>
    );
  }
}

export const ConfirmationModal = ConfirmationModalComponent as React.ComponentClass<ConfirmationModalProps>;
