import * as React from "react";

import { inject, observer } from "mobx-react";

import { ModalCloser } from "aegis/functionality/components/modal-closer";
import { ModalProps } from "aegis/stores";
import {
  AlignContent,
  AlignItems,
  Background,
  Button,
  Color,
  CoreText,
  Display,
  JustifyContent,
  Layout,
  StyledLayout,
  TextType
} from "twitch-core-ui";

import "./styles.scss";

export const NotificationModalConfirmSelector = "notification-modal.confirm";

export interface NotificationModalProps {
  title: string;
  body: string;
  error?: boolean;
  disabled?: boolean;
  onConfirm?: () => void;
}

type Props = NotificationModalProps & ModalProps;

@inject("modal")
@observer
export class NotificationModalComponent extends React.Component<Props, {}> {
  confirm = () => {
    if (this.props.onConfirm) {
      this.props.onConfirm();
    } else {
      this.props.modal!.closeModal();
    }
  };

  render() {
    const { title, body, disabled } = this.props;
    return (
      <StyledLayout
        className="confirmation-modal"
        background={!!this.props.error ? Background.Overlay : Background.Accent}
        padding={{ top: 2, left: 2, bottom: 2, right: 2 }}
        alignContent={AlignContent.Center}
        alignItems={AlignItems.Center}
      >
        <ModalCloser closeModal={this.confirm} />

        <StyledLayout
          background={Background.Base}
          fullHeight={true}
          alignContent={AlignContent.Center}
          padding={{ top: 5 }}
        >
          <CoreText type={TextType.H5} color={Color.Alt}>
            {title}
          </CoreText>
          <StyledLayout borderTop borderBottom padding={{ y: 1 }} margin={{ y: 1 }}>
            <CoreText color={!!this.props.error ? Color.Error : Color.Base}>{body}</CoreText>
          </StyledLayout>
          <Layout display={Display.Flex} justifyContent={JustifyContent.Center}>
            <Button
              data-track-click="notification-modal-close"
              onClick={this.confirm}
              disabled={!!disabled}
              data-test-selector={NotificationModalConfirmSelector}
            >
              {"Close"}
            </Button>
          </Layout>
        </StyledLayout>
      </StyledLayout>
    );
  }
}

export const NotificationModal = NotificationModalComponent as React.ComponentClass<NotificationModalProps>;
