import * as React from "react";

import { mount } from "enzyme";
import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { BACKDROP_TEST_SELECTOR, CLOSE_BUTTON_TEST_SELECTOR, PortalModal } from "./component";

Enzyme.configure({ adapter: new Adapter() });

describe("Portal Modal", () => {
  let wrapper: Enzyme.ReactWrapper;

  const modalRoot = document.createElement("div");
  modalRoot.setAttribute("id", "portal-modal-root");
  const body = document.querySelector("body");
  if (body) {
    body.appendChild(modalRoot);
  }

  afterEach(() => {
    if (wrapper.length > 0) {
      wrapper.unmount();
    }
  });

  it("renders in the PortalModalRoot", () => {
    wrapper = mount(
      <PortalModal onClose={jest.fn()}>
        <div>"test"</div>
      </PortalModal>
    );
    expect(wrapper.hostNodes()).toMatchSnapshot();
  });

  it("calls passed onClose prop on clicking close button", () => {
    const onClose = jest.fn();

    wrapper = mount(
      <PortalModal onClose={onClose}>
        <div>"test"</div>
      </PortalModal>
    );

    expect(onClose).toHaveBeenCalledTimes(0);
    wrapper.find({ "data-test-selector": CLOSE_BUTTON_TEST_SELECTOR }).simulate("click");
    expect(onClose).toHaveBeenCalledTimes(1);
  });

  it("calls onClose on click out when onClickOut is true", () => {
    const onClose = jest.fn();

    wrapper = mount(
      <PortalModal onClose={onClose} closeOnClickOut>
        <div>"test"</div>
      </PortalModal>
    );
    expect(onClose).toHaveBeenCalledTimes(0);
    wrapper.find({ "data-test-selector": BACKDROP_TEST_SELECTOR }).simulate("click");
    expect(onClose).toHaveBeenCalledTimes(1);
  });

  it("doesn't call onClose on click out when onClickOut is false", () => {
    const onClose = jest.fn();

    wrapper = mount(
      <PortalModal onClose={onClose}>
        <div>"test"</div>
      </PortalModal>
    );
    expect(onClose).toHaveBeenCalledTimes(0);
    wrapper.find({ "data-test-selector": BACKDROP_TEST_SELECTOR }).simulate("click");
    expect(onClose).toHaveBeenCalledTimes(0);
  });

  it("fails to render without a PortalModalRoot", () => {
    spyOn(console, "error");
    let error;

    body!.removeChild(modalRoot);

    try {
      wrapper = mount(
        <div className="app-tree">
          <PortalModal onClose={jest.fn()}>
            <div>"test"</div>
          </PortalModal>
        </div>
      );
    } catch (e) {
      error = e;
    }
    expect(error).toBeInstanceOf(Error);
    expect(error).toMatchSnapshot();
  });
});
