import * as React from "react";

import { HotKeys } from "react-hotkeys";

import { InjectLayout, SVG, SVGAsset } from "twitch-core-ui";
import { PortalInject } from "./components/portal-inject";

import "./styles.scss";

export const CLOSE_BUTTON_TEST_SELECTOR = "portal-modal__close-button";
export const BACKDROP_TEST_SELECTOR = "portal-modal__backdrop";

interface Props {
  /**
   * Called when the "x" button on the top-right is clicked, on
   * clickout, or on escape key press.
   * When called, parent is expected to unmount this component
   * Without this, modal will not be possible to close
   */
  onClose: () => void;
  /**
   * Set closeOnClickOut to "true" if you would like clicking on the
   * grey area outside of the modal to close the modal
   */
  closeOnClickOut?: boolean;
  /**
   * Set overrideWidth to "true" if you would like to specify a width
   * for the modal other than the default specified in styles.scss
   */
  overrideWidth?: boolean;
}

/**
 * This component is used to mount a modal within a component outside of
 * the modal's DOM tree. While the component will be "mounted" in the
 * render tree within the ReactVirtualDOM, a portal will be created to
 * the <PortalRoot>, wherever it is mounted in the document
 */
export class PortalModal extends React.Component<Props> {
  private hotkeyAreaRef: HTMLElement | null = null;
  private hotkeyMap = { closeModal: "esc" };
  private hotkeyHandlers = {
    closeModal: () => {
      this.props.onClose();
    }
  };

  public componentDidMount() {
    this.focusOnHotkeyArea();
  }

  public render() {
    const childClassName = this.props.overrideWidth ? undefined : "default-width-modal";
    const body = (
      <InjectLayout fullWidth className="modal-hotkey-container">
        <HotKeys keyMap={this.hotkeyMap} handlers={this.hotkeyHandlers}>
          <div className="modal-body" tabIndex={0} ref={ref => (this.hotkeyAreaRef = ref)}>
            <div className="close-button" onClick={this.onCloseClick} data-test-selector={CLOSE_BUTTON_TEST_SELECTOR}>
              <SVG asset={SVGAsset.Close} />
            </div>
            <div className={childClassName}>{this.props.children}</div>
          </div>
        </HotKeys>
      </InjectLayout>
    );

    return (
      <PortalInject>
        <div
          className="modal-with-portal combo-breaker"
          onClick={this.backgroundClicked}
          data-test-selector={BACKDROP_TEST_SELECTOR}
        >
          {body}
        </div>
      </PortalInject>
    );
  }

  private onCloseClick = (e: React.MouseEvent) => {
    e.stopPropagation();
    this.props.onClose();
  };

  private focusOnHotkeyArea = () => {
    if (this.hotkeyAreaRef) {
      this.hotkeyAreaRef.focus();
    }
  };

  private backgroundClicked = (event: React.MouseEvent) => {
    if (event.target instanceof Element && event.target.classList.contains("modal-with-portal")) {
      if (this.props.closeOnClickOut) {
        this.props.onClose();
      } else {
        this.focusOnHotkeyArea();
      }
    }
  };
}
