import * as React from "react";

import { autorun, observable, observe } from "mobx";
import { inject, observer } from "mobx-react";

import { PortalModal } from "aegis/functionality/components/portal-modal";
import { StackedModalStore } from "aegis/stores";
import { ModalHeader } from "./modal-header";

import "./styles.scss";

export interface Props {
  /**
   * Called when the "x" button on the top-right is clicked, on
   * clickout, or on escape key press.
   * When called, parent is expected to unmount this component
   * Without this, modal will not be possible to close
   */
  onClose?: () => void;
  /**
   * Title to be rendered in header
   */
  title?: string;
  /**
   * If true, ModalHeader will not be rendered
   */
  hideHeader?: boolean;

  /**
   * Store, auto-populated by Mobx
   */
  stackedModalStore?: StackedModalStore;
}

/**
 * Creates a stackable <PortalModal>. Always visible when rendered,
 * "onClose" should unmount.
 *
 * The most recently rendered <StackedModal> will be visible, with the
 * rest living "underneath". <StackedModal> can be rendered within
 * another <StackedModal> and will appear in order of render. Header is
 * optional and can be removed via "hideHeader" prop
 */
@inject("stackedModalStore")
@observer
export class StackedModal extends React.Component<Props> {
  @observable private child: JSX.Element | null = null;

  public componentDidMount() {
    const children = React.Children.toArray(this.props.children) as React.ReactElement<{}>[];
    if (children.length !== 1) {
      throw new Error("StackedModal needs a child to render");
    }
    if (!this.props.stackedModalStore) {
      throw new Error("No StackedModalStore...?");
      return;
    }
    this.child = children[0];
    this.props.stackedModalStore.openModal(this.child);
    autorun(() => {
      observe(this.props.stackedModalStore!, () => null);
    });
  }

  public componentWillUnmount() {
    this.props.stackedModalStore!.closeModal(this.child!);
  }

  public render() {
    const modalHeader = (
      <ModalHeader
        title={this.props.title}
        onClose={this.props.stackedModalStore!.isOnlyModal(this.child!) ? undefined : this.props.onClose!}
      />
    );
    if (this.props.stackedModalStore!.shouldRender(this.child!)) {
      return (
        <PortalModal onClose={this.props.onClose!}>
          {this.props.hideHeader ? null : modalHeader}
          <div className={this.props.hideHeader ? "modal-body-no-header" : "modal-body-with-header"}>
            {this.props.children}
          </div>
        </PortalModal>
      );
    }
    // Assures that children of this modal will still be rendered, even
    // if this modal is not visible
    return <div style={{ display: "none" }}>{this.props.children}</div>;
  }
}
