import * as React from "react";

import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { aegisUploadsRegex, TextWithLinks } from "./component";

Enzyme.configure({ adapter: new Adapter() });

describe("TextWithLinks", () => {
  it("creates links in the passed message", () => {
    const validLink = "https://s3-us-west-2.amazonaws.com/aegis-uploads/corgi";
    const message = "Here's some text with a link! " + validLink;
    const wrapper = Enzyme.mount(<TextWithLinks>{message}</TextWithLinks>);
    expect(wrapper.html().includes(`<a href="${validLink}" target="blank">`)).toBeTruthy();
  });

  it("renders a script tag as escaped text", () => {
    const fakeScript = '<script>alert("This would be bad")</script>';
    const fakeScriptEscaped = '&lt;script&gt;alert("This would be bad")&lt;/script&gt;';

    const message = "Here's some text with a link! " + fakeScript;
    const messageEscaped = "Here's some text with a link! " + fakeScriptEscaped;

    const wrapper = Enzyme.mount(<TextWithLinks>{message}</TextWithLinks>);

    expect(wrapper.html().includes(messageEscaped)).toBeTruthy();
    expect(wrapper.find("script").length).toBe(0);
  });

  describe("linking regex", () => {
    it("catches aegis file upload links correctly", () => {
      const goodLinks = [
        "https://s3-us-west-2.amazonaws.com/aegis-uploads-1234/images/corgi.png",
        "s3-us-west-2.amazonaws.com/aegis-uploads-1234/",
        "s3-us-west-2.amazonaws.com/file-uploads/"
      ];
      goodLinks.forEach(link => {
        expect(aegisUploadsRegex.test(link)).toBeTruthy();
      });
    });

    it("catches leviathan file upload links correctly", () => {
      const goodLinks = [
        "https://leviathan-prod.s3-us-west-2.amazonaws.com/file-uploads/BpVLvoMA12D2UbTz/runescapertl--2018-08-27.png"
      ];
      goodLinks.forEach(link => {
        expect(aegisUploadsRegex.test(link)).toBeTruthy();
      });
    });

    it("doesn't catch links it shouldn't", () => {
      const badLinks = [
        "http://www.somewhere-bad.com/s3-us-west-2.amazonaws.com/aegis-uploads-1234/",
        "www.google.com",
        "s3-us-west-2.amazonaws.com/",
        "s3-us-west-2.amazonaws.com.hacked.com/aegis-uploads"
      ];
      badLinks.forEach(link => {
        expect(aegisUploadsRegex.test(link)).toBeFalsy();
      });
    });
  });
});
