import * as React from "react";

import { newGuid } from "aegis/functionality/utils/unique-id";
import { Balloon, BalloonProps, Display, Layout, Position, Tooltip, TooltipProps } from "twitch-core-ui";

export const MOUSE_ENTER_DETECTOR_SELECTOR = "toggle-balloon-wrapper__mouse-enter-detector";

export interface PublicProps {
  /** specifies that the balloon's content should always exist in the DOM, regardless of whether or not it is visible.
   * needed for current channel-header resizing implementation.
   */
  alwaysMountBalloonContent?: boolean;
  display?: Display;
  /** specifies whether balloon starts in shown state when initially rendered. */
  openByDefault?: boolean;
  onToggle?: (isClosed: boolean) => void;
  tooltipProps?: TooltipProps;
}

interface State {
  /** Prop passed to underlying Balloon to indicidate whether it should hide or show via CSS. */
  showBalloon: boolean;
  /** 'Sticky' state begins false; can only be set to true by mouseenter, click, or programmatic toggle via public toggleBallon method */
  hasInteracted: boolean;
}

type Props = PublicProps;

/**
 * Container for a click-to-toggle balloon that will close when
 * the user clicks outside. It should contain exactly two child elements:
 * - A button or link first, its onClick prop will be overwritten
 * - A core-ui Balloon element second, its show and toggle props will be over-written.
 * Doesn't mount the child Balloon unless alwaysMountBalloonContent prop is true, or user has
 * moused-over or clicked the corresponding Button.
 */
export class ToggleBalloonWrapper extends React.Component<Props, State> {
  private toggleBalloonId = newGuid();
  public state = {
    showBalloon: !!this.props.openByDefault,
    hasInteracted: !!this.props.openByDefault
  };

  public componentDidMount() {
    if (this.props.openByDefault) {
      document.addEventListener("click", this.handleGlobalClick, true);
    }
  }

  public componentWillUpdate(_nextProps: {}, nextState: State) {
    if (this.state.showBalloon !== nextState.showBalloon) {
      if (nextState.showBalloon) {
        document.addEventListener("click", this.handleGlobalClick, true);
      } else {
        document.removeEventListener("click", this.handleGlobalClick, true);
      }
    }
  }

  public componentWillUnmount() {
    document.removeEventListener("click", this.handleGlobalClick, true);
  }

  public render() {
    // tslint:disable-next-line:no-any
    const children = React.Children.toArray(this.props.children) as React.ReactElement<any>[];
    if (children.length !== 2) {
      throw new Error("ToggleBalloonWrapper should only be given two children: a clickable and a Balloon");
    }

    const [originalButton, originalBalloon] = children;
    if (originalBalloon.type !== Balloon) {
      throw new Error("ToggleBalloonWrapper needs a Balloon as its second child element");
    }

    const button = React.cloneElement(originalButton, {
      onClick: this.handleButtonClick
    });

    let balloon: React.ReactElement<BalloonProps> | null = null;

    if (this.state.hasInteracted || this.props.alwaysMountBalloonContent) {
      balloon = React.cloneElement(originalBalloon, {
        show: this.state.showBalloon
      });
    }

    let buttonElement = button;

    if (this.props.tooltipProps) {
      buttonElement = <Tooltip {...this.props.tooltipProps}>{button}</Tooltip>;
    }

    return (
      <Layout data-toggle-balloon-id={this.toggleBalloonId} display={this.props.display} position={Position.Relative}>
        <div
          style={{ display: "inherit" }}
          onMouseEnter={this.onMouseEnter}
          data-test-selector={MOUSE_ENTER_DETECTOR_SELECTOR}
        >
          {buttonElement}
        </div>
        {balloon}
      </Layout>
    );
  }

  public toggleBalloon(newValue?: boolean) {
    if (newValue === undefined) {
      newValue = !this.state.showBalloon;
    }

    this.setState({
      showBalloon: newValue,
      hasInteracted: true
    });
  }

  private handleButtonClick = (e: MouseEvent) => {
    e.stopPropagation();
    this.setState(
      prevState => ({
        showBalloon: !prevState.showBalloon,
        hasInteracted: true
      }),
      () => {
        if (this.props.onToggle) {
          this.props.onToggle(!this.state.showBalloon);
        }
      }
    );
  };

  private handleGlobalClick = (e: MouseEvent) => {
    const target = e.target as Element;
    if (!target.matches(`[data-toggle-balloon-id="${this.toggleBalloonId}"] *`)) {
      this.setState(
        {
          showBalloon: false
        },
        () => {
          if (this.props.onToggle) {
            this.props.onToggle(!this.state.showBalloon);
          }
        }
      );
    }
  };

  private onMouseEnter = () => {
    if (!this.state.hasInteracted) {
      this.setState({ hasInteracted: true });
    }
  };
}
