import * as React from "react";

import { User } from "aegis/features/user";
import * as Models from "aegis/models";
import { Color, CoreText, Display, Layout, SearchInput, SegmentedButton, SegmentedButtonOption } from "twitch-core-ui";

import "./styles.scss";

export interface ComponentProps {
  lookupByID: (id: string) => void;
  lookupByLogin: (login: string) => void;
  currentUser?: Models.User;
  error?: string;
}

export interface PublicProps {
  defaultID?: string;
  defaultLogin?: string;
  hideUser?: boolean;

  // name must be unique among all rendered instances of this on the page
  name: string;
}

interface State {
  loginSelected: boolean;
  value: string;
}

type Props = ComponentProps & PublicProps;

export enum TestSelectors {
  InputField = "user-input-text-field",
  User = "user-input-user",
  IDButton = "user-input-id-button"
}

export class UserInputComponent extends React.Component<Props, State> {
  public constructor(props: Props) {
    super(props);

    if (props.defaultLogin) {
      this.state = { loginSelected: true, value: props.defaultLogin };
    } else if (props.defaultID) {
      this.state = { loginSelected: false, value: props.defaultID };
    } else {
      this.state = { loginSelected: true, value: "" };
    }
  }

  public componentDidMount() {
    // Force fetch if no currentUser despite defaultLogin and defaultID
    if (!this.props.currentUser) {
      if (this.props.defaultLogin) {
        this.props.lookupByLogin(this.props.defaultLogin);
      } else if (this.props.defaultID) {
        this.props.lookupByID(this.props.defaultID);
      }
    }
  }

  public render() {
    const { name } = this.props;
    const search = (
      <Layout display={Display.Flex}>
        <Layout padding={{ right: 1 }}>
          <SearchInput
            value={this.state.value}
            placeholder={"Login/User ID..."}
            onChange={this.updateValue}
            onBlur={this.submit}
            onKeyDown={this.onKeyDown}
            data-test-selector={TestSelectors.InputField}
          />
        </Layout>
        <Layout>
          <SegmentedButton>
            <SegmentedButtonOption
              label={"Login"}
              name={name}
              checked={this.state.loginSelected}
              value={"login"}
              onChange={this.selectType}
            />
            <SegmentedButtonOption
              label={"ID"}
              name={name}
              checked={!this.state.loginSelected}
              value={"id"}
              onChange={this.selectType}
              data-test-selector={TestSelectors.IDButton}
            />
          </SegmentedButton>
        </Layout>
      </Layout>
    );

    if (this.props.hideUser) {
      return search;
    }

    const { currentUser, error } = this.props;
    let extraInfo: JSX.Element | null = null;
    if (currentUser) {
      extraInfo = <User userID={currentUser.id} showStatus showType inline data-test-selector={TestSelectors.User} />;
    }
    if (this.state.value !== "" && error) {
      extraInfo = <CoreText color={Color.Error}>{error}</CoreText>;
    }

    return (
      <Layout>
        {search}
        <Layout padding={{ top: 0.5 }} className={"user-lookup-result-space"}>
          {extraInfo}
        </Layout>
      </Layout>
    );
  }

  private updateValue: React.FormEventHandler<HTMLInputElement> = e => {
    this.setState({ value: e.currentTarget.value }, () => {
      // Edge case for clicking (x) button in <SearchInput /> not firing submission
      if (this.state.value === "") {
        this.submit();
      }
    });
  };

  private selectType = (e: React.FormEvent<HTMLInputElement>) => {
    this.setState({ loginSelected: e.currentTarget.value === "login" }, () => {
      this.submit();
    });
  };

  private submit = () => {
    if (this.state.loginSelected) {
      this.props.lookupByLogin(this.state.value);
    } else {
      this.props.lookupByID(this.state.value);
    }
  };

  private onKeyDown = (e: React.KeyboardEvent<HTMLInputElement>) => {
    if (e.key === "Enter") {
      this.submit();
    }
  };
}
