import * as React from "react";

import ApolloClient, { ApolloQueryResult } from "apollo-client";
import gql from "graphql-tag";
import { compose, withApollo } from "react-apollo";

import * as Fragments from "aegis/fragments";
import * as Models from "aegis/models";
import { PublicProps, UserInputComponent } from "./component";

export type Response = {
  user: Models.User;
};

export const FETCH_USER_QUERY = gql`
  query user($id: ID, $login: String) {
    user(id: $id, login: $login) {
      ...UserFields
    }
  }
  ${Fragments.UserFields}
`;

type InputVariables = {
  login?: string;
  id?: string;
};

interface ContainerProps {
  onChange?: (user?: Models.User) => void;
}

interface ApolloProps {
  client: ApolloClient<{}>;
}

export type Props = ContainerProps & PublicProps & ApolloProps;

interface State {
  currentUser?: Models.User;
  error?: string;
}

export class UserQueryWrapper extends React.Component<Props, State> {
  state: State = {};

  public render() {
    return (
      <UserInputComponent
        lookupByID={this.lookupByID}
        lookupByLogin={this.lookupByLogin}
        currentUser={this.state.currentUser}
        error={this.state.error}
        {...this.props}
      />
    );
  }

  private lookupByLogin = (login: string) => {
    this.fireQuery({ login: login });
  };

  private lookupByID = (id: string) => {
    this.fireQuery({ id: id });
  };

  private fireQuery = ({ id, login }: InputVariables) => {
    // Handle empty string submission (set currentUser undefined and update parent)
    if ((login === "" && !id) || (id === "" && !login)) {
      this.changeUser(undefined);
      return;
    }

    this.props.client
      .query({
        query: FETCH_USER_QUERY,
        variables: { id: id, login: login }
      })
      .then(({ data }: ApolloQueryResult<Response>) => {
        if (data && data.user) {
          this.changeUser(data.user);
        } else {
          this.changeUser(undefined, "Unable to find user");
        }
      })
      .catch(() => {
        this.changeUser(undefined, "Unable to find user");
      });
  };

  private changeUser = (user?: Models.User, error?: string) => {
    if (!user || !this.state.currentUser || this.state.currentUser.id !== user.id) {
      this.setState({ currentUser: user, error: error }, () => {
        if (this.props.onChange) {
          this.props.onChange(this.state.currentUser);
        }
      });
    }
  };
}

export const UserInput: React.ComponentClass<PublicProps & ContainerProps> = compose(withApollo)(UserQueryWrapper);
