import * as React from "react";

import { copyTextToClipboard } from "aegis/functionality/utils/contrib/twilight/copy-text-to-clipboard";
import { ApolloError } from "apollo-client";
import { toast } from "react-toastify";
import { InFeatureNotification, InFeatureNotificationType } from "twitch-core-ui";

import "./styles.scss";

export interface ErrorNotificationOptions {
  error: string | ApolloError;
  copyErrorToClipboard?: boolean;
}

function getCopyLinkProps(options?: ErrorNotificationOptions) {
  if (!options || !options.copyErrorToClipboard) {
    return undefined;
  }
  const { error } = options;
  const messageForClipboard: string = typeof error === "string" ? error : error.message;
  return {
    onClick: () => copyTextToClipboard(messageForClipboard),
    children: "Click to copy error to clipboard"
  };
}

/**
 * Wrapper for react-toastify's "toast.error", handles console logging and copy-to-clipboard logic.
 * @param {string} notificationMessage - The message to be displayed in the notification
 * @param {ErrorNotificationOptions} options - (Optional) An `options` object with `error` and `copyErrorToClipboard` flags
 *
 * TODO: Add tracking for errors, this is a good place to do it
 */
export function errorNotification(notificationMessage: string, options?: ErrorNotificationOptions) {
  const linkProps = getCopyLinkProps(options);
  const description = options && !options.copyErrorToClipboard ? options.error.toString() : "";

  toast.error(
    <InFeatureNotification
      title={notificationMessage}
      description={description}
      link={linkProps}
      type={InFeatureNotificationType.Error}
    />,
    {
      toastId: description,
      autoClose: 8000,
      className: "custom-error-notifications",
      closeButton: false,
      progressClassName: "custom-error-progress-bar"
    }
  );
}

/**
 * Function to create a generic Apollo "error handler". Optionally takes in a title string
 * @param notificationMessage - (Optional) The message to be displayed in the notification's header
 */
export function apolloErrorHandler(notificationMessage?: string) {
  return function(e: ApolloError) {
    const errorMessage = notificationMessage ? notificationMessage : "Error";
    errorNotification(errorMessage, { error: e, copyErrorToClipboard: true });
  };
}
