// tslint:disable:max-classes-per-file
import * as React from "react";

/**
 * Use this when creating a static displayName field in your HOC/wrapped component classes.
 */
export function createHOCDisplayName<P>(name: string, Component: React.ComponentType<P>) {
  return `${name}(${Component.displayName || Component.name || "Component"})`;
}

/**
 * Wrapper for React.createContext
 * Returns a provider component and a HOC component creator to use on consuming components
 * @param name Used to give generated components relevant displayName values
 * @param fallbackState The context value given to consumers that get mounted without a provider parent
 */
export function createContext<Context extends object>(name: string, fallbackState: Context) {
  // tslint:disable-next-line:ban
  const FullContext = React.createContext<Context>(fallbackState);
  const { Consumer, Provider } = FullContext;

  /**
   * Passthrough component to avoid blowing up shallow renders until enzyme doesn't support providers.
   * After enzyme updates this could probably just pass the real native provider back
   */
  class InnerProvider extends React.Component<{ value: Context }> {
    public static displayName = `Inner${name}Provider`;

    public componentDidMount() {
      const contextType = typeof this.props.value;
      if (contextType !== "object" || !this.props.value) {
        console.error(
          new Error(`${InnerProvider.displayName} rendered without an appropriate initial value`),
          "Set an initial value in your provider component at mount time."
        );
      }
    }

    public render() {
      return <Provider value={this.props.value}>{this.props.children}</Provider>;
    }
  }

  /**
   * Consumer HOC for accessing this test context
   */
  function withContext<ProvidedProps, P = {}>(mapContextToProps: (c: Context, ownProps: P) => ProvidedProps) {
    return (WrappedComponent: React.ComponentType<P>) => {
      return class WithHOCContextConsumer extends React.Component<P> {
        public static displayName = createHOCDisplayName(`With${name}`, WrappedComponent);
        public render() {
          return (
            <Consumer>
              {value => {
                const injectedProps = mapContextToProps(value, this.props);
                return <WrappedComponent {...this.props} {...injectedProps} />;
              }}
            </Consumer>
          );
        }
      };
    };
  }

  return { withContext, InnerProvider, FullContext };
}
