// Imported from Twilight version 5.61.0
import { LocationDescriptorObject } from "history";
import { default as createBrowserHistory } from "history/createBrowserHistory";
import { default as createMemoryHistory } from "history/createMemoryHistory";

/**
 * Creates a browser history object that uses replace on push if the path haven't changed.
 */
export function createHistory(useMemoryHistory?: boolean) {
  const history = useMemoryHistory ? createMemoryHistory() : createBrowserHistory();
  const historyPush: typeof history.push = history.push.bind(history);
  const matcher = /^([^#?]+)?(\?[^#]+)?(\#.+)?$/i;

  history.push = (pathOrLocation: string | LocationDescriptorObject, state?: {}) => {
    let pathname = history.location.pathname;
    let search = "";
    let hash = "";

    if (typeof pathOrLocation === "string") {
      const matches = matcher.exec(pathOrLocation);
      if (matches) {
        if (matches[1]) {
          pathname = matches[1];
        }

        if (matches[2]) {
          search = matches[2];
        }

        if (matches[3]) {
          hash = matches[3];
        }
      } else {
        // Should be impossible, but better safe than sorry.
        historyPush(pathOrLocation, state);
        return;
      }
    } else {
      if (pathOrLocation.pathname) {
        pathname = pathOrLocation.pathname;
      }

      if (pathOrLocation.search) {
        search = pathOrLocation.search;
        if (!search.startsWith("?")) {
          search = "?" + search;
        }
      }

      if (pathOrLocation.hash) {
        hash = pathOrLocation.hash;
        if (!hash.startsWith("#")) {
          hash = "#" + hash;
        }
      }
    }

    // `any` casts below are a workaround for TypeScript being confused about the signature for push & replace.
    if (
      (pathname === history.location.pathname ||
        (!pathname.startsWith("/") && history.location.pathname.endsWith("/" + pathname))) &&
      search === history.location.search &&
      hash === history.location.hash
    ) {
      // tslint:disable-next-line:no-any
      history.replace(pathOrLocation as any, state);
    } else {
      // tslint:disable-next-line:no-any
      historyPush(pathOrLocation as any, state);
    }
  };

  return history;
}
