import * as React from "react";

import { mount } from "enzyme";
import * as Enzyme from "enzyme";
import * as Adapter from "enzyme-adapter-react-16";

import { getFirstAncestorWithClassOrAttribute } from "./get-keys";

Enzyme.configure({ adapter: new Adapter() });

describe("getFirstAncestorWithClassOrAttribute", () => {
  it("grabs the correct parent elements when the class exists", () => {
    const someNestedHTML: JSX.Element = (
      <div className="parent">
        <div>
          <div className="child">Test text</div>
        </div>
      </div>
    );

    const wrapper = mount(someNestedHTML);
    const child = wrapper.find(".child");
    const parent = getFirstAncestorWithClassOrAttribute(child.getDOMNode(), "parent");
    expect(parent!.className).toBe("parent");
  });

  it("grabs the correct parent elements when the attribute exists", () => {
    const someNestedHTML: JSX.Element = (
      <div data-key-selector="anything">
        <div>
          <div className="child">Test text</div>
        </div>
      </div>
    );

    const wrapper = mount(someNestedHTML);
    const child = wrapper.find(".child");
    const parent = getFirstAncestorWithClassOrAttribute(child.getDOMNode(), "data-key-selector");
    expect(parent!.getAttribute("data-key-selector")).toBe("anything");
  });

  it("grabs the correct parent elements prioritizing class over attribute", () => {
    const someNestedHTML: JSX.Element = (
      <div className="wrong-parent" data-key-selector="wrong">
        <div className="data-key-selector" data-parent="herpderp">
          <div>
            <div className="child">Test text</div>
          </div>
        </div>
      </div>
    );

    const wrapper = mount(someNestedHTML);
    const child = wrapper.find(".child");
    const parent = getFirstAncestorWithClassOrAttribute(child.getDOMNode(), "data-key-selector");
    expect(parent!.getAttribute("data-key-selector")).toBeNull();
    expect(parent!.className).toBe("data-key-selector");
    expect(parent!.getAttribute("data-parent")).toBe("herpderp");
  });

  it("returns null when the parent does not exist", () => {
    const someNestedHTML: JSX.Element = (
      <div className="notaparent">
        <div>
          <div className="child">Test text</div>
        </div>
      </div>
    );

    const wrapper = mount(someNestedHTML);
    const child = wrapper.find(".child");
    const parent = getFirstAncestorWithClassOrAttribute(child.getDOMNode(), "parent");
    expect(parent).toBeNull();
  });

  it("grabs the correct parent elements when parent has >1 class", () => {
    const someNestedHTML: JSX.Element = (
      <div className="parent withmore classes becauseitcan">
        <div>
          <div className="child">Test text</div>
        </div>
      </div>
    );

    const wrapper = mount(someNestedHTML);
    const child = wrapper.find(".child");
    const parent = getFirstAncestorWithClassOrAttribute(child.getDOMNode(), "parent");
    expect(parent!.className).toBe("parent withmore classes becauseitcan");
  });
});
