/*
  getKeysOfSelection is used to get data-key values from a specific
  ancestor in a Selection
  i.e. getting row indexes from a user selection grabbed via
  window.getSelection()

  Note:
    Due to this util's use of Selections from window.getSelection(), it is
    not easily testable. JSDOM does not support getSelection() calls:
    https://github.com/jsdom/jsdom/issues/317

    If you make changes to this file, please manually test the following
    components that use it:
    - Chatlog (for custom copy-paste formatting on selected rows)
*/
export function getKeysOfSelection(
  selection: Selection,
  classOrAttributeName: string
): { first: number; last: number } {
  const firstRow = getFirstAncestorWithClassOrAttribute(selection.anchorNode, classOrAttributeName);
  const lastRow = getFirstAncestorWithClassOrAttribute(selection.focusNode, classOrAttributeName);
  const first = firstRow ? parseInt(firstRow.getAttribute("data-key")!, 10) : 0;
  const last = lastRow ? parseInt(lastRow.getAttribute("data-key")!, 10) : null;

  // Catch if any elements aren't found
  if (first === null || last === null) {
    return { first: -1, last: -1 };
  }

  // Catch if the selection is within a single element
  if (selection.anchorNode.parentElement === selection.focusNode.parentElement) {
    return { first, last };
  }

  if (first > last) {
    return { first: last, last: first + 1 };
  }
  return { first, last: last + 1 };
}

export function getFirstAncestorWithClassOrAttribute(element: Node, classOrAttributeName: string) {
  while (element.parentElement) {
    // Allow for "separated" elements to break the tree, escaping this flow
    // Example: Modals is opened from chat log row, copying from modal should not copy from row
    if (element.parentElement.classList.contains("combo-breaker")) {
      return null;
    }
    if (
      element.parentElement.classList.contains(classOrAttributeName) ||
      element.parentElement.getAttribute(classOrAttributeName)
    ) {
      return element.parentElement;
    }
    element = element.parentElement;
  }
  return null;
}

export function hasClassInAncestry(element: Node, classname: string) {
  while (element.parentElement) {
    if (element.parentElement.classList.contains(classname)) {
      return true;
    }
    element = element.parentElement;
  }
  return false;
}
