import * as puppeteer from "puppeteer";

const stagingEndpoint = "https://aegis.service.safety-staging.justin.tv/";

// Set a default value to make typescript happy
// Actual values are set in scripts/test.js
const testHost = process.env.TEST_HOST ? process.env.TEST_HOST : "";
// const testServer = process.env.TEST_SERVER ? process.env.TEST_SERVER : ""; // Commented out for lint. This is
const testDomain = process.env.TEST_DOMAIN ? process.env.TEST_DOMAIN : "";
const userName = process.env.AEGIS_TEST_USER_NAME ? process.env.AEGIS_TEST_USER_NAME : "";
const password = process.env.AEGIS_TEST_USER_PASSWORD ? process.env.AEGIS_TEST_USER_PASSWORD : "";
const defaultTimeout = 10000;

const defaultArgs = {
  args: ["--no-sandbox", "--disable-setuid-sandbox"],
  slowMo: 10,
  headless: process.env.KARMA_ENV !== "debug",
  defaultViewport: {
    width: 1100,
    height: 600
  }
};

const launchWithURL = async (url: string, cookies?: puppeteer.SetCookie[]) => {
  browser = await puppeteer.launch(defaultArgs);
  mainPage = await browser.newPage();
  if (cookies) {
    await mainPage.setCookie(...cookies);
  }
  await mainPage.goto(url);
};

// Commented out for lint. This function should be used
// when retrieving information from test server

// type ErrorResponse = {
//   status: number;
//   message: string;
//   error: string;
// };

// const fetchJsonFromURL = async (url: string) => {
//   return new Promise((resolve, reject) => {
//     const xhr = new XMLHttpRequest();
//     xhr.open("GET", url);
//     xhr.onreadystatechange = function() {
//       if (xhr.readyState === xhr.DONE) {
//         if (xhr.status === 200) {
//           const parsedResponse = JSON.parse(xhr.responseText);
//           resolve(parsedResponse);
//         } else {
//           try {
//             const error = JSON.parse(xhr.responseText).message as ErrorResponse;
//             reject(error);
//           } catch (e) {
//             console.error("Failed to parse error response, error is ", e);
//             console.error("Error Message: ", xhr.responseText);
//             reject(e);
//           }
//         }
//       }
//     };
//     xhr.send();
//   });
// };

let browser: puppeteer.Browser;
let mainPage: puppeteer.Page;

describe("on page load", () => {
  beforeAll(() => {
    if (!userName) {
      fail("User name is not specified");
    }
  });

  describe("Pre-authed load", () => {
    beforeAll(async () => {
      await launchWithURL(stagingEndpoint);
      const buttonText = await mainPage.$eval(".parent .tw-button .tw-button__text", e => e.innerHTML);
      expect(buttonText).toBe("Log in to Aegis using Twitch Guardian SSO service");

      const targetPromise = new Promise<puppeteer.Target>(resolve => browser.once("targetcreated", resolve));
      await mainPage.click(".tw-button");
      const target = await targetPromise;
      const loginPage = await target.page();

      const guardianFormText = await loginPage.$eval("h4", e => e.innerHTML);
      expect(guardianFormText).toBe("Authorize With LDAP");

      await loginPage.type("input[name=username]", userName);
      await loginPage.type("input[name=password]", password);

      const authButton = await loginPage.waitFor("button[id=accept-oauth]");
      authButton.click();

      await mainPage.waitFor(".App");
      const cookies = await mainPage.cookies();

      cookies.forEach(cookie => {
        cookie.domain = testDomain;
      }); // The version of Chromium we are using doesn't support .domain. need to specify full path for cookie to work

      mainPage = await browser.newPage();
      await mainPage.setCookie(...cookies);
      await mainPage.goto(testHost);

      await mainPage.waitFor(".App");
    }, 15000);

    afterAll(() => {
      if (browser) {
        browser.close();
      }
    });

    test(
      "chatlog page loads",
      async () => {
        // Puppeteer's .click doesn't work with how core-ui implemented dropdown nav
        // Running document.querySelector("[data-test-selector='top-nav__chatlog']").click();
        // works on browser but not with puppeteer
        // https://github.com/GoogleChrome/puppeteer/issues/3535 <--- probably the issue
        await mainPage.$eval("[data-test-selector='top-nav__chatlog']", e => (e as HTMLButtonElement).click());

        await mainPage.waitFor("[data-test-selector='chatlog-page']");
      },
      defaultTimeout
    );

    test(
      "banform page loads",
      async () => {
        await mainPage.$eval("[data-test-selector='top-nav__ban-form']", e => (e as HTMLButtonElement).click());

        await mainPage.waitFor("[data-test-selector='ban-form-page']");
      },
      defaultTimeout
    );

    test(
      "alt lookup page loads",
      async () => {
        await mainPage.$eval("[data-test-selector='top-nav__alt-lookup']", e => (e as HTMLButtonElement).click());

        await mainPage.waitFor("[data-test-selector='alt-lookup__page']");
      },
      defaultTimeout
    );

    test(
      "clip mass deleter page loads",
      async () => {
        await mainPage.$eval("[data-test-selector='top-nav__clip-mass-deleter']", e =>
          (e as HTMLButtonElement).click()
        );

        await mainPage.waitFor("[data-test-selector='clips-mass-deleter__page']");
      },
      defaultTimeout
    );

    test(
      "chat activity page loads",
      async () => {
        await mainPage.$eval("[data-test-selector='top-nav__chat-activity']", e => (e as HTMLButtonElement).click());

        await mainPage.waitFor("[data-test-selector='chat-activity__page']");
      },
      defaultTimeout
    );
  });
});
