import { SVGAsset } from "twitch-core-ui";
import { BanException, Stream, Suspension } from ".";

export type User = {
  createdAt?: string;
  displayName: string;
  id: string;
  isBanned: boolean;
  isDMCABanned: boolean;
  isDeleted: boolean;
  language: string;
  login: string;
  profileImage: string;
  roles?: UserRole[];
  streamEmbedURL: string;
  __typename: string;
};

export interface DetailedUser extends User {
  acceptableNameConfidence: number;
  banException?: BanException;
  bannedUntil?: string;
  channel?: Channel;
  dmcaViolationCount: number;
  isLiveStreaming: boolean;
  reportedCount: number;
  warnedCount: number;
  reportSentCount: number;
  stream?: Stream;
  suspension?: Suspension;
  tosViolationCount: number;
  isEscalatePartner: boolean;
}

export interface Channel {
  game?: string;
  language: string;
  profileBannerImage?: string;
  offlineImage?: string;
  title: string;
  mature: boolean;
}

export enum UserRole {
  Staff = "STAFF",
  Admin = "ADMIN",
  CEO = "CEO",
  EXEC = "EXEC",
  Partner = "PARTNER",
  Affiliate = "AFFILIATE",
  Developer = "DEVELOPER",
  VerifiedBot = "VERIFIED_BOT"
}

export const UserRoleTagMapping = new Map<UserRole, { text: string; icon: SVGAsset }>([
  [UserRole.Staff, { text: "Staff", icon: SVGAsset.Wrench }],
  [UserRole.Admin, { text: "Admin", icon: SVGAsset.Automod }],
  [UserRole.CEO, { text: "Ceo", icon: SVGAsset.Trophy }],
  [UserRole.EXEC, { text: "Exec", icon: SVGAsset.Trophy }],
  [UserRole.Partner, { text: "Partner", icon: SVGAsset.Verified }],
  [UserRole.Affiliate, { text: "Affiliate", icon: SVGAsset.Heart }],
  [UserRole.Developer, { text: "Developer", icon: SVGAsset.Experiment }],
  [UserRole.VerifiedBot, { text: "Verified Bot", icon: SVGAsset.Android }]
]);
// getNameAsDisplayed returns the name as displayed on Twitch
export function getNameAsDisplayed(user: User): string {
  if (!user || !user.displayName || !user.login) {
    return "UNKNOWN";
  }
  if (user.displayName.toLowerCase() === user.login.toLowerCase()) {
    return user.displayName;
  }
  return `${user.displayName} (${user.login})`;
}

// channelLink returns a link to user's channel page
export function channelLink(user: User): string {
  return `https://www.twitch.tv/${user.login}?tmi_host=false`;
}

// accountInfoLink returns a link to user's account info page
export function accountInfoLink(user: User): string {
  return `/accountinfo?id=${user.id}`;
}
