import * as React from "react";

import gql from "graphql-tag";

import { StrikeButton } from "aegis/features/strike-button";
import { User } from "aegis/features/user";
import * as Fragments from "aegis/fragments";
import { Query } from "aegis/functionality/apollo-wrapper";
import { AltUserInfo } from "aegis/models";
import { BalloonDirection, ButtonType, Color, CoreText, LoadingSpinner } from "twitch-core-ui";

export const ALT_USER_QUERY = gql`
  query altUsers($id: ID!) {
    altUsers(id: $id) {
      type
      user {
        ...UserFields
        createdAt
      }
      matched {
        ...UserFields
        createdAt
      }
    }
  }
  ${Fragments.UserFields}
`;

interface Data {
  altUsers: Array<AltUserInfo>;
}

interface Variables {
  id: string;
}

class AllAltsQuery extends Query<Data, Variables> {}

export const AltViewer = (props: { id: string }) => (
  <AllAltsQuery displayName="AltViewer" query={ALT_USER_QUERY} variables={{ id: props.id }}>
    {({ loading, error, data }) => {
      if (loading) {
        return <LoadingSpinner />;
      }

      if (error) {
        console.error(error);
        return <CoreText color={Color.Error}>An error occurred while loading alt users: {error.message}</CoreText>;
      }

      return (
        <table style={{ width: "100%" }}>
          <thead>
            <tr>
              <th>Login</th>
              <th>Type</th>
              <th>Matched User</th>
              <th>Created On</th>
              <th>Actions</th>
            </tr>
          </thead>
          <tbody>
            {data!.altUsers.map(function(altUser: AltUserInfo, i: number) {
              const { type, user, matched } = altUser;
              return (
                user && (
                  <tr key={`alt-user-${i}`} style={{ outline: "thin solid" }}>
                    <td>
                      <User
                        user={user}
                        inline={true}
                        showStatus={true}
                        showType={true}
                        balloonDirection={BalloonDirection.BottomLeft}
                      />
                    </td>
                    <td>{type}</td>
                    <td>
                      <User user={matched} inline={true} showStatus={true} showType={true} />
                    </td>
                    <td>{user.createdAt}</td>
                    <td>
                      <StrikeButton
                        type={ButtonType.Hollow}
                        user={user}
                        banFormProps={{
                          targetUserID: user.id,
                          targetUser: user.login
                        }}
                      />
                    </td>
                  </tr>
                )
              );
            })}
          </tbody>
        </table>
      );
    }}
  </AllAltsQuery>
);
