import * as React from "react";

import { action, observable, reaction } from "mobx";
import { observer, Provider } from "mobx-react";
import * as QueryString from "query-string";
import { compose, withApollo } from "react-apollo";
import { Helmet } from "react-helmet";
import { RouteComponentProps } from "react-router";

import { AltLookupTable } from "aegis/features/alt-lookup-table";
import { LinkToChatActivity } from "aegis/features/alt-lookup-table/link-to-chat-activity";
import { UserInput } from "aegis/functionality/components/user-input";
import { User } from "aegis/models";
import { AltLookUpStore } from "aegis/stores/altLookUpStore";

import { AlignContent, Display, FlexDirection, Layout } from "twitch-core-ui";

import "./styles.scss";

interface QueryProps {
  id?: string;
  login?: string;
}

type Props = RouteComponentProps<QueryProps>;

export enum TestSelectors {
  Page = "alt-lookup__page"
}

@observer
class AltUserLookupComponent extends React.Component<Props> {
  private updateHistoryDisposer = reaction(
    () => this.userID,
    userID => {
      const params = QueryString.stringify({ id: userID });
      this.props.history!.push({ search: params, pathname: `/altlookup` });
    }
  );

  private altLookUpStore = new AltLookUpStore();
  @observable private userID?: string;
  readonly defaultUserID?: string;
  readonly defaultUserLogin?: string;

  constructor(props: Props) {
    super(props);
    const { search } = this.props.location;
    const queryProps: QueryProps = QueryString.parse(search) as QueryProps;
    this.userID = this.defaultUserID = queryProps.id;
    this.defaultUserLogin = queryProps.login;
  }

  componentWillUnmount() {
    this.updateHistoryDisposer();
  }

  render() {
    return (
      <Provider altLookUpStore={this.altLookUpStore}>
        <Layout
          padding={2}
          className="alt-lookup"
          data-test-selector={TestSelectors.Page}
          alignContent={AlignContent.Center}
        >
          <Helmet>
            <title>Alt User Lookup</title>
          </Helmet>
          <Layout display={Display.InlineFlex} flexDirection={FlexDirection.Row} fullWidth>
            <Layout flexGrow={4}>
              <UserInput
                name="altlookup-user"
                onChange={this.onChange}
                defaultID={this.defaultUserID}
                defaultLogin={this.defaultUserLogin}
              />
            </Layout>
            {this.userID && <LinkToChatActivity />}
          </Layout>
          <AltLookupTable id={this.userID} />
        </Layout>
      </Provider>
    );
  }

  @action
  private onChange = (user?: User) => {
    this.userID = user ? user.id : undefined;
  };
}

export const AltUserLookup = compose(withApollo)(AltUserLookupComponent);
