import * as React from "react";

import { ApolloClient } from "apollo-client";
import gql from "graphql-tag";
import { observable } from "mobx";
import { observer } from "mobx-react";
import { compose, graphql, GraphqlQueryControls, withApollo } from "react-apollo";
import { Helmet } from "react-helmet";

import { WithConfirmationModal } from "aegis/functionality/components/confirmation-modal";
import {
  Background,
  Button,
  Color,
  CoreLink,
  CoreText,
  Display,
  JustifyContent,
  Layout,
  StyledLayout,
  Table,
  TableBody,
  TableCell,
  TableHeader,
  TableHeading,
  TableRow,
  TextArea,
  TextType
} from "twitch-core-ui";

export enum TestSelectors {
  Page = "clips-mass-deleter__page"
}

interface PublicProps {}

export interface ApolloProps {
  client?: ApolloClient<{}>;
}

export interface DeleteClipsResponse {
  deleteClips: {
    success: boolean[];
  };
}

interface GraphQLProps {
  data: DeleteClipsResponse & GraphqlQueryControls;
}

export type Props = PublicProps & GraphQLProps & ApolloProps;

// Clips slugs on the clips mini site are prefixed with https://clips.twitch.tv/
const clipsSiteSlugRegex = /https:\/\/clips\.twitch\.tv\/([a-z0-9_]+)/gi;
// Clips slugs on the main site are prefixed with https://wwww.twitch.tv/<channelname>/clip/
const twitchSiteSlugRegex = /https:\/\/www\.twitch\.tv\/.*\/clip\/([a-z0-9_]+)/gi;

export const deleteClips = gql`
  mutation deleteClips($slugs: DeleteClipsInput!) {
    deleteClips(input: $slugs) {
      success
    }
  }
`;

export interface DeletionResults {
  slug: string;
  success: boolean;
}

const DeleterResults: React.SFC<{ results: DeletionResults[] }> = props => {
  if (props.results.length === 0) {
    return null;
  }

  const rows: JSX.Element[] = [];
  props.results.forEach((result, _) => {
    const url = `https://clips.twitch.tv/${result.slug}`;
    rows.push(
      <TableRow key={result.slug}>
        <TableCell>
          <CoreLink to={url}>{url}</CoreLink>
        </TableCell>
        <TableCell>{result.slug}</TableCell>
        <TableCell>
          <CoreText>{result.success ? "Deleted" : "Not Deleted"}</CoreText>
        </TableCell>
      </TableRow>
    );
  });

  return (
    <Table>
      <TableHeader>
        <TableHeading label="URL" />
        <TableHeading label="Slug" />
        <TableHeading label="Status" />
      </TableHeader>
      <TableBody>{rows}</TableBody>
    </Table>
  );
};

@observer
class ClipsMassDeleterComponent extends React.Component<Props> {
  @observable private contents: string = "";
  @observable private deleting: boolean = false;
  @observable private results: DeletionResults[] = [];
  @observable private error: string | null = null;

  public render() {
    return (
      <Layout padding={2} data-test-selector={TestSelectors.Page}>
        <Helmet>
          <title>Clips mass deleter</title>
        </Helmet>

        <StyledLayout border padding={3} background={Background.Alt2}>
          <CoreText type={TextType.H2}>Enter clips to be deleted</CoreText>
          <Layout margin={{ y: 1 }}>
            <TextArea rows={20} onChange={this.contentChanged} />
          </Layout>
          <Layout margin={{ y: 1 }} display={Display.Flex} justifyContent={JustifyContent.End}>
            <WithConfirmationModal
              title="Mass Delete Clips"
              body="Are you sure you want to delete all of these clips?"
              confirmLabelOverride="Delete"
              onConfirm={this.deleteClips}
            >
              {confirm => {
                return (
                  <Button data-track-click="clips-mass-deleter-submit" onClick={confirm} disabled={this.deleting}>
                    Delete Clips
                  </Button>
                );
              }}
            </WithConfirmationModal>
          </Layout>
          {this.error ? <CoreText color={Color.Error}>{this.error}</CoreText> : null}
          <DeleterResults results={this.results} />
        </StyledLayout>
      </Layout>
    );
  }

  private contentChanged: React.FormEventHandler<HTMLTextAreaElement> = e => {
    this.error = null;
    this.contents = e.currentTarget.value;
  };

  private deleteClips: () => void = async () => {
    try {
      this.deleting = true;

      const slugs: string[] = [];
      let result: RegExpExecArray | null;
      while ((result = clipsSiteSlugRegex.exec(this.contents)) !== null) {
        slugs.push(result[1]);
      }
      while ((result = twitchSiteSlugRegex.exec(this.contents)) !== null) {
        slugs.push(result[1]);
      }
      if (slugs.length === 0) {
        this.error = "No clip slugs found!";
        return;
      }

      const results = await this.props.client!.mutate<DeleteClipsResponse>({
        mutation: deleteClips,
        variables: {
          slugs: {
            slugs: slugs
          }
        }
      });

      this.results = slugs.map(function(slug: string, i: number) {
        return {
          slug: slug,
          success: results!.data!.deleteClips.success[i]
        };
      });
    } catch (error) {
      this.error = JSON.stringify(error);
    } finally {
      this.deleting = false;
    }
  };
}

export const ClipsMassDeleter = compose(
  withApollo,
  graphql<DeleteClipsResponse, PublicProps>(deleteClips)
)(ClipsMassDeleterComponent);
