import * as React from "react";

import * as QueryString from "query-string";
import { Helmet } from "react-helmet";
import { RouteComponentProps } from "react-router";

import { GameDirectory } from "aegis/features/game-directory";
import {
  AlignItems,
  Button,
  ButtonType,
  Color,
  CoreText,
  Display,
  FlexDirection,
  Input,
  InputType,
  JustifyContent,
  Layout,
  TextType
} from "twitch-core-ui";

export interface QueryProps {
  game?: string;
  grayscale?: string;
}

export class GameDirectoryPage extends React.Component<RouteComponentProps<QueryProps>> {
  render() {
    const { location } = this.props;
    let queryProps: QueryProps = {};
    if (location.search) {
      queryProps = QueryString.parse(location.search) as QueryProps;
    }

    const grayscale = queryProps.grayscale === "true";

    return (
      <>
        <Helmet>
          <title>Game Directory</title>
        </Helmet>
        <Layout
          padding={{ top: 1 }}
          display={Display.Flex}
          justifyContent={JustifyContent.Center}
          alignItems={AlignItems.Center}
          flexDirection={FlexDirection.Column}
        >
          <Layout display={Display.Flex} alignItems={AlignItems.Center}>
            <Layout padding={{ right: 0.5 }}>{"Select Game: "}</Layout>
            <Input
              type={InputType.Text}
              onBlur={e => {
                this.changeGame(e.currentTarget.value);
              }}
              onKeyDown={e => {
                if (e.key === "Enter") {
                  this.changeGame(e.currentTarget.value);
                }
              }}
              defaultValue={queryProps.game}
            />
            <Layout padding={{ left: 1 }}>
              <Button
                blurAfterClick
                onClick={() => this.toggleGrayScale(!grayscale)}
                type={grayscale ? ButtonType.Hollow : ButtonType.Default}
              >
                Toggle Grayscale
              </Button>
            </Layout>
          </Layout>
          <Layout display={Display.Flex} alignItems={AlignItems.Center}>
            <CoreText type={TextType.Span} color={Color.Alt2}>
              Use "no-game" to look up streams with no game selected
            </CoreText>
          </Layout>
        </Layout>
        <GameDirectory game={queryProps.game} grayscale={grayscale} />
      </>
    );
  }

  private changeGame = (game: string) => {
    const queryParams = QueryString.parse(location.search) as QueryProps;
    const updatedParams = QueryString.stringify({ ...queryParams, game: game });
    this.props.history.push({ pathname: "/gamedirectory", search: updatedParams });
  };

  private toggleGrayScale = (grayscale: boolean) => {
    const queryParams = QueryString.parse(location.search) as QueryProps;
    const updatedParams = QueryString.stringify({ ...queryParams, grayscale: grayscale });
    this.props.history.push({ pathname: "/gamedirectory", search: updatedParams });
  };
}
