import * as React from "react";

import { inject, observer } from "mobx-react";
import * as QueryString from "query-string";

import { AuthStore } from "aegis/stores";
import { AlignContent, AlignItems, Button, Color, CoreText, Display, FlexDirection, Layout } from "twitch-core-ui";

import "./styles.scss";

interface AuthProps {
  authStore?: AuthStore;
}

interface QueryProps {
  query?: string;
}

type Props = QueryProps & AuthProps;

class GuardianResponse {
  code: string;
  state: string;
}

class State {
  errorMessage: string;
  authLink: string;
}

class AuthURL {
  location: string;
}

const MessageEventType = "message";

@inject("authStore")
@observer
export class LoginImposerContainer extends React.Component<Props, State> {
  constructor(props: Props, context: {}) {
    super(props, context);

    this.state = {
      errorMessage: "",
      authLink: ""
    };
  }

  public render() {
    const {
      state,
      props: { authStore, children }
    } = this;
    if (!authStore) {
      return;
    }
    if (authStore.hasToken) {
      return children;
    }

    const { pathname, search } = window.location;
    let g = new GuardianResponse();
    if (pathname === "/oauth") {
      g = QueryString.parse(search) as GuardianResponse;
      this.notifyAndClose(g);
    }

    if (!state.authLink) {
      this.fetchAuthLink();
    }

    return (
      <Layout
        className="parent"
        display={Display.Flex}
        alignContent={AlignContent.Center}
        alignItems={AlignItems.Center}
        flexDirection={FlexDirection.Column}
      >
        <Button onClick={this.openAuthPage} disabled={!state.authLink}>
          Log in to Aegis using Twitch Guardian SSO service
        </Button>
        <CoreText color={Color.Error}>{authStore.errorMessage ? authStore.errorMessage : state.errorMessage}</CoreText>
      </Layout>
    );
  }

  private openAuthPage = async (e: React.MouseEvent<HTMLElement>) => {
    const { state } = this;

    this.setState({
      errorMessage: ""
    });

    e.preventDefault();
    window.open(state.authLink, "Login", "height=420,width=840");
    window.addEventListener(MessageEventType, this.handleCode);
  };

  private fetchAuthLink = async () => {
    const response = await fetch("/auth_redirect", { method: "GET" });

    if (response.ok) {
      try {
        const authURL: AuthURL = await response.json();
        this.setState({
          authLink: authURL.location
        });
      } catch (error) {
        console.error("Failed to parse auth link response ", error);
      }
    } else {
      console.error("Failed to fetch auth link. Server returned invalid response", response);
    }
  };

  private handleCode = async (e: MessageEvent) => {
    if (e.origin !== window.location.origin) {
      console.error("Received message from different origin", e);
      this.setState({
        errorMessage: `Received message from a different origin.
        Event origin: "${e.origin}"; Current origin: "${window.location.origin}"`
      });

      return;
    }

    const response = e.data as GuardianResponse;
    if (!response.code) {
      this.setState({
        errorMessage: "Server did not return a code. Please try again"
      });

      return;
    }

    if (!response.state) {
      this.setState({
        errorMessage: "Server did not return a state. Please try again"
      });

      return;
    }

    try {
      const serverResponse = await fetch(`/auth_exchange?code=${response.code}&state=${response.state}`, {
        method: "GET"
      });
      if (serverResponse.ok && this.props.authStore) {
        const token = await serverResponse.json();

        this.props.authStore.tokenReceived(token.access_token);
      } else {
        this.setState({ errorMessage: "Unexpected server response code " + serverResponse.status });
      }
    } catch (error) {
      console.error("Failed to obtain token", error);
      this.setState({
        errorMessage: "Failed to obtain token. Please try again."
      });
    }
  };

  private notifyAndClose = (response: GuardianResponse) => {
    if (!window.opener) {
      console.error("Parent window not found! Doing nothing");

      return;
    }

    try {
      // noop operation to test if the request is able to access parent window.
      // if it has no access (most likely due to cross origin), this call would error out
      // tslint:disable-next-line:no-unused-expression
      window.opener.location.origin;
    } catch (error) {
      // Avoid triggering an infinite state update if error message is already set
      if (this.state.errorMessage) {
        return;
      }

      console.error("Current window has a different origin than parent", error);
      this.setState({
        errorMessage: `Current window has a different origin than parent.
        Are you using the right url (http vs https?)`
      });

      return;
    }

    window.opener.postMessage(response, window.location.origin);
    window.close();
  };
}

export const LoginImposer = LoginImposerContainer as React.ComponentClass<{}>;
