import { Admin } from "aegis/models";
import * as Cookies from "js-cookie";
import { action, autorun, computed, observable } from "mobx";

const authTokenKey = "aegis-auth-token";

export interface AuthStoreProps {
  authStore?: AuthStore;
}

export class AuthStore {
  @observable private _token: string | null = null;
  @observable private _loggedInUser: Admin | null = null;
  @observable private _errorMessage: string | null = null;

  constructor() {
    this._token = Cookies.get(authTokenKey)!;

    if (!this._token) {
      this._token = null;
    }

    autorun(() => {
      let domain = window.location.hostname;
      if (domain.indexOf(".") !== -1) {
        domain = domain.substring(domain.indexOf("."));
      }
      if (!this._token) {
        Cookies.remove(authTokenKey, {
          domain: domain
        });
      } else {
        Cookies.set(authTokenKey, this._token, {
          domain: domain,
          path: "/",
          expires: 30 // Expires after 30 days
        });
      }
    });
  }

  @computed
  get hasToken(): boolean {
    return !!this._token;
  }

  @computed
  get token(): string | null {
    return this._token;
  }

  @computed
  get hasLoggedInUser(): boolean {
    return !!this._loggedInUser;
  }

  @computed
  get loggedInUser(): Admin | null {
    return this._loggedInUser;
  }

  @action
  tokenReceived = (token: string): void => {
    this._token = token;
    this._errorMessage = null;
  };

  @action
  userLogin = (user: Admin): void => {
    this._loggedInUser = user;
  };

  @action
  reset = (errorMessage?: string | null): void => {
    this._token = null;
    this._errorMessage = errorMessage ? errorMessage : null;
    this._loggedInUser = null;
  };

  @computed
  get errorMessage(): string | null {
    return this._errorMessage;
  }
}
