import { ChatLogsInput } from "aegis/features/chatlogs";
import { action, computed, observable } from "mobx";

export type ChatStoreParams = {
  first?: string;
  userids: string;
  channelids: string;
  after: string;
  before: string;
  regex: string;
};

export class ChatStore {
  @observable private _userID: string | undefined;
  @observable private _channelIDs: string | undefined;
  @observable private _first = 500;
  @observable private _after = Date.now() - 7 * 84600 * 1000; // 7 days ago
  @observable private _before = Date.now();
  @observable private _regex: string | null = null;

  @observable private _trigger: boolean = false;

  @computed
  get userID(): string | undefined {
    return this._userID;
  }

  @computed
  get channelID(): string | undefined {
    return this._channelIDs;
  }

  @computed
  get first(): number {
    return this._first;
  }

  @computed
  get after(): number {
    return this._after;
  }

  @computed
  get before(): number {
    return this._before;
  }

  @computed
  get regex(): string | null {
    return this._regex;
  }

  @computed
  get trigger(): boolean {
    return this._trigger;
  }

  @computed
  get params(): ChatStoreParams {
    return {
      first: this.first.toString(),
      userids: this.userID || "",
      channelids: this.channelID || "",
      after: this.after.toString(),
      before: this.before.toString(),
      regex: this.regex!
    };
  }

  @computed
  get paramsAsInput(): ChatLogsInput {
    return {
      first: this.first,
      user: this.userID,
      channel: this.channelID,
      timeStart: this.after.toString(),
      timeEnd: this.before.toString(),
      regex: this.regex
    };
  }

  @action
  updateUserIDs = (id?: string): void => {
    this._userID = id;
  };

  @action
  updateChannelID = (id?: string): void => {
    this._channelIDs = id;
  };

  @action
  updateFirst = (first: number): void => {
    this._first = first;
  };

  @action
  updateAfter = (after: number): void => {
    this._after = after;
  };

  @action
  updateBefore = (before: number): void => {
    this._before = before;
  };

  @action
  updateRegex = (regex: string | null): void => {
    if (!regex) {
      regex = null;
    }
    this._regex = regex;
  };

  @action
  update = (params: ChatStoreParams): void => {
    const { first, userids, channelids, after, before, regex } = params;
    let firstNum = this._first;
    if (first) {
      firstNum = parseInt(first, 10);
      if (isNaN(firstNum)) {
        firstNum = this._first;
      }
    }

    this.updateUserIDs(userids);
    this.updateChannelID(channelids);
    this.updateFirst(firstNum);
    this.updateAfter(after ? parseInt(after, 10) : 0);
    this.updateBefore(before ? parseInt(before, 10) : Date.now());
    this.updateRegex(regex ? regex : "");
    this._trigger = !this._trigger;
  };
}
