import { action, observable } from "mobx";

export const storageKey = "aegis:deferredEmotes";
export const ttlMillis = 8 * 60 * 60 * 1000; // 8 hours

export interface DeferredEmoteStoreProps {
  deferredEmoteStore?: DeferredEmoteStore;
}

export type DeferredEmotes = {
  updatedAt?: string;
  ids?: string[];
};

export class DeferredEmoteStore {
  @observable private _deferredEmotes: Set<string> = new Set();

  constructor() {
    const storeKeys = localStorage.getItem(storageKey);

    if (storeKeys) {
      try {
        const storedEmotes = JSON.parse(storeKeys) as DeferredEmotes;
        if (
          storedEmotes.updatedAt &&
          storedEmotes.ids &&
          // Verify the keys we expect there is still there against corrupted data
          new Date(storedEmotes.updatedAt).getTime() + ttlMillis > new Date().getTime()
        ) {
          storedEmotes.ids.forEach(id => this._deferredEmotes.add(id));
        }
      } catch (error) {
        console.error("Error loading deferred emotes. Ignoring", error);
        console.error("Local storage value", storeKeys);
      }
    }

    // Reset store to refresh the timer/remove entries expired
    this.updateStorage();
  }

  isDeferred(id: string) {
    return this._deferredEmotes.has(id);
  }

  @action
  updateDeferred(deferredIDs: string[]) {
    deferredIDs.forEach(id => this._deferredEmotes.add(id));

    this.updateStorage();
  }

  private updateStorage() {
    const ids: string[] = [];
    this._deferredEmotes.forEach(id => {
      ids.push(id);
    });

    const storedEmotes: DeferredEmotes = {
      updatedAt: new Date().toISOString(),
      ids: ids
    };

    localStorage.setItem(storageKey, JSON.stringify(storedEmotes));
  }
}
