import { action, computed, observable } from "mobx";

export interface ReportStoreProps {
  reportStore?: ReportStore;
}

export class ReportStore {
  @observable private _focusedID: string | undefined = undefined;
  @observable private _expandedIDs: Set<string> = new Set();
  @observable private _visibleReportsIDs: Set<string> = new Set();

  @computed
  get focusedID(): string | undefined {
    return this._focusedID;
  }

  @computed
  get hasFocusedID(): boolean {
    return !!this._focusedID;
  }

  @computed
  get expandedIDs(): Set<string> {
    return this._expandedIDs;
  }

  @computed
  get visibleReportsIDs(): Set<string> {
    return this._visibleReportsIDs;
  }

  @action
  setFocusedID(id: string) {
    this._focusedID = id;
  }

  @action
  setReportVisibile(id: string) {
    this._visibleReportsIDs.add(id);
  }

  @action
  setReportNotVisible(id: string) {
    this._visibleReportsIDs.delete(id);
  }

  isExpanded(id: string): boolean {
    return this._expandedIDs.has(id);
  }

  @action
  expand(id: string) {
    this._expandedIDs.add(id);
  }

  @action
  collapse(id: string) {
    this._expandedIDs.delete(id);
  }

  @action
  toggleExpanded(id: string) {
    if (this._expandedIDs.has(id)) {
      this._expandedIDs.delete(id);
    } else {
      this._expandedIDs.add(id);
    }
  }

  @action
  removeReport(id: string) {
    this.collapse(id);
    this.setReportNotVisible(id);

    if (id === this._focusedID) {
      this._focusedID = undefined;
    }
  }

  @action
  clear() {
    this._expandedIDs.clear();
    this._visibleReportsIDs.clear();
    this._focusedID = undefined;
  }
}
