import * as React from "react";

import { InMemoryCache, NormalizedCacheObject } from "apollo-cache-inmemory";
import { ApolloClient } from "apollo-client";
import { FetchResult, GraphQLRequest } from "apollo-link";
import { mount, MountRendererProps, ReactWrapper } from "enzyme";
import { Provider } from "mobx-react";
import { ReactElement } from "react";
import { GraphqlQueryControls } from "react-apollo";
import { MockedProvider, MockedResponse as ReactApolloMockedResponse, mockSingleLink } from "react-apollo/test-utils";
import * as renderer from "react-test-renderer";
import { TestRendererOptions } from "react-test-renderer";

import * as stores from "aegis/stores";

/**
 * Helper for testing elements that rely on GraphQL data.
 * Uses react-apollo <MockedProvider> HOC to provide data props that
 * mock those provided by react-apollo Provider HOC.
 * Applies the passed-in testRenderer function to the <MockedProvider>-wrapped JSX Element and returns a promise,
 * which resolves with a rendered version of the passed-in element.  Depending on the mock provided, this component
 * will reflect all data props loaded, or an error, as if the first request had just completed (networkStatus 7 or 8).
 */

type AnyRenderOptions = MountRendererProps | TestRendererOptions;

// tslint:disable-next-line:no-any
function safeToUpdate(wrapper: any, testRenderer: Function) {
  return wrapper.update && testRenderer === mount;
}

function getGQLRenderable(
  element: JSX.Element,
  mocks: MockedResponse[],
  testRenderer: Function,
  renderOptions?: AnyRenderOptions
) {
  jest.useFakeTimers();

  const { wrappedComponent, client } = wrapWithGQLProvider(element, mocks);

  const wrapper = testRenderer(wrappedComponent, renderOptions);

  jest.runOnlyPendingTimers();
  jest.useRealTimers();

  return {
    wrapper,
    client
  };
}

function withGraphQLData<T>(
  element: JSX.Element,
  mocks: MockedResponse[],
  testRenderer: Function,
  renderOptions?: AnyRenderOptions
) {
  const { wrapper } = getGQLRenderable(element, mocks, testRenderer, renderOptions);

  return new Promise<T>(resolve => {
    setImmediate(() => {
      if (safeToUpdate(wrapper, testRenderer)) {
        wrapper.update();
      }
      resolve(wrapper);
    });
  });
}

interface WithGQLElementAndClient<T> {
  wrapper: T;
  client: ApolloClient<NormalizedCacheObject>;
}

function withGraphQLDataAndClient<T>(
  element: JSX.Element,
  mocks: MockedResponse[],
  testRenderer: Function,
  renderOptions?: AnyRenderOptions
) {
  const { wrapper, client } = getGQLRenderable(element, mocks, testRenderer, renderOptions);

  return new Promise<WithGQLElementAndClient<T>>(resolve => {
    setImmediate(() => {
      if (safeToUpdate(wrapper, testRenderer)) {
        wrapper.update();
      }
      resolve({
        wrapper,
        client
      });
    });
  });
}

// The MockedResponse in react-apollo/test-utils has been updated to require the newUserData property.
// None of the mocks we have written has this field so rather than update all the existing mocks
// for no reason, I decided to proxy the actual interface with this backwards compatible version.
export interface MockedResponse {
  request: GraphQLRequest;
  result?: FetchResult;
  error?: Error;
  delay?: number;
  newUserData?: () => GraphQLRequest;
}

export function getNewMockClient(mocks?: MockedResponse[]) {
  return new ApolloClient({
    // See the comment at the top of the file regarding why this blind cast is necessary
    link: mockSingleLink(...(mocks ? (mocks as ReactApolloMockedResponse[]) : [])),
    cache: new InMemoryCache()
  });
}

/**
 * Usable on its own if you want to mount a GQL decorated component and check its loading state
 */
export function wrapWithGQLProvider(element: JSX.Element, mocks: MockedResponse[]) {
  const client = new ApolloClient({
    // See the comment at the top of the file regarding why this blind cast is necessary
    link: mockSingleLink(...(mocks as ReactApolloMockedResponse[])),
    cache: new InMemoryCache()
  });

  return {
    wrappedComponent: (
      <MockedProvider mocks={mocks}>
        <Provider {...stores} client={client}>
          {element}
        </Provider>
      </MockedProvider>
    ),
    client
  };
}

/**
 * Uses our GraphQL mocking helper to return an enzyme mounted wrapper with mocked
 * GraphQL responses applied.
 */
export function mountWithGraphQLData<P>(
  element: ReactElement<P>,
  mocks: MockedResponse[],
  renderOptions?: MountRendererProps
) {
  return withGraphQLData<ReactWrapper<P, {}>>(element, mocks, mount, renderOptions);
}

/**
 * Uses our GraphQL mocking helper to return an enzyme mounted wrapper with mocked
 * GraphQL responses applied as well as the apollo client to influence the store in
 * tests as you please.
 */
export function mountWithGraphQLDataAndClient<P>(
  element: ReactElement<P>,
  mocks: MockedResponse[],
  renderOptions?: MountRendererProps
) {
  return withGraphQLDataAndClient<ReactWrapper<P, {}>>(element, mocks, mount, renderOptions);
}

/**
 * Uses our GraphQL mocking helper to return markup from react-test-renderer with mocked
 * GraphQL responses applied.
 */
export function renderWithGraphQLData<P>(
  element: ReactElement<P>,
  mocks: MockedResponse[],
  renderOptions?: TestRendererOptions
) {
  return withGraphQLData<renderer.ReactTestRenderer>(element, mocks, renderer.create, renderOptions);
}

/**
 * Useful for mocking GraphQL's "data" prop in shallow render testing.
 */
export function mockGraphQLDataProps<Data>(
  queryResponse: Data,
  dataProps?: Partial<GraphqlQueryControls>
): GraphqlQueryControls & Data {
  return Object.assign(
    {
      networkStatus: 7,
      loading: false,
      error: undefined,
      variables: {},
      fetchMore: jest.fn(),
      refetch: jest.fn(),
      startPolling: jest.fn(),
      stopPolling: jest.fn(),
      subscribeToMore: jest.fn(),
      updateQuery: jest.fn()
    },
    queryResponse,
    dataProps
  );
}
