package spade

import (
	"bytes"
	"context"
	"encoding/base64"
	"encoding/json"
	"expvar"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"sync/atomic"

	"code.justin.tv/hygienic/errors"
)

// Event represents a spade event
type Event struct {
	Name string `json:"event"`
	// Properties should be a JSON serializable struct
	Properties interface{} `json:"properties"`
}

func encodeBatchOfEvents(events ...Event) (string, error) {
	jb, err := json.Marshal(events)
	if err != nil {
		return "", err
	}

	jb64 := base64.URLEncoding.EncodeToString(jb)
	return jb64, nil
}

// HTTPRequestDoer can execute http requests
type HTTPRequestDoer interface {
	// Do should be like http.Client.Do
	Do(req *http.Request) (*http.Response, error)
}

var _ HTTPRequestDoer = &http.Client{}

// Logger outputs values
type Logger interface {
	Log(vals ...interface{})
}

// StatSender sends stats
type StatSender interface {
	Inc(string, int64, float32) error
}

// Client sends data to spade
type Client struct {
	Config     Config
	HTTPClient HTTPRequestDoer `nilcheck:"nodepth"`
	Logger     Logger
	Statter    StatSender

	Backlog chan Event

	closeSignal  chan struct{}
	doneDraining chan struct{}

	stats struct {
		TotalEventsSent int64
		AsyncErrors     int64
	}
}

// Vars allows client to expose expvar info
func (c *Client) Vars() expvar.Var {
	return expvar.Func(func() interface{} {
		return map[string]interface{}{
			"backlog_size":      len(c.Backlog),
			"total_events_sent": atomic.LoadInt64(&c.stats.TotalEventsSent),
			"async_errors":      atomic.LoadInt64(&c.stats.AsyncErrors),
		}
	})
}

// Setup should be called before you use the client
func (c *Client) Setup() error {
	c.doneDraining = make(chan struct{})
	c.closeSignal = make(chan struct{})
	if c.Backlog == nil {
		c.Backlog = make(chan Event, c.Config.GetBacklogSize())
	}
	if c.HTTPClient == nil {
		c.HTTPClient = http.DefaultClient
	}
	return nil
}

func (c *Client) log(vals ...interface{}) {
	if c.Logger != nil {
		c.Logger.Log(vals...)
	}
}

func (c *Client) incStat(stat string, value int64, rate float32) {
	if c.Statter != nil {
		err := c.Statter.Inc(stat, value, rate)
		if err != nil {
			c.log("err", "failure sending stats", "staterr", err)
		}
	}
}

// Start should run in the background to drain the client's queue
func (c *Client) Start() error {
	defer func() {
		close(c.doneDraining)
	}()
	for {
		eventsToSend := make([]Event, 1, c.Config.GetBatchSize())
		select {
		case eventsToSend[0] = <-c.Backlog:
		case <-c.closeSignal:
			return nil
		}

	outerLoop:
		for int64(len(eventsToSend)) < c.Config.GetBatchSize() {
			select {
			case anotherEvent := <-c.Backlog:
				eventsToSend = append(eventsToSend, anotherEvent)
			default:
				break outerLoop
			}
		}
		ctx := context.Background()
		ctx, cancel := context.WithTimeout(ctx, c.Config.GetTimeout())
		if err := c.BlockingSendEvents(ctx, eventsToSend...); err != nil {
			c.incStat("errors.send_fail", int64(len(eventsToSend)), 1.)
			atomic.AddInt64(&c.stats.AsyncErrors, 1)
			c.log("err", err, "failed to drain spade queue")
		}
		cancel()
	}
}

// Close ends the client
func (c *Client) Close() error {
	close(c.closeSignal)
	<-c.doneDraining
	return nil
}

// VerifyEndpoint checks that spade works
func (c *Client) VerifyEndpoint(ctx context.Context) error {
	req, err := http.NewRequest("GET", c.Config.GetSpadeHost()+"/xarth", nil)
	if err != nil {
		return err
	}
	req = req.WithContext(ctx)
	resp, err := c.HTTPClient.Do(req)
	if err != nil {
		return err
	}
	defer func() {
		if e := resp.Body.Close(); e != nil {
			c.log("err", err, "failed to close response body")
		}
	}()
	if resp.StatusCode != http.StatusOK {
		return errors.Errorf("invalid status code %d", resp.StatusCode)
	}
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return err
	}
	if string(body) != "XARTH" {
		return errors.Errorf("invalid body %s", string(body))
	}
	return nil
}

// QueueEvents adds events to the internal queue.  Returns instantly if the queue is full.
func (c *Client) QueueEvents(events ...Event) {
	for _, event := range events {
		select {
		case c.Backlog <- event:
		default:
			c.incStat("errors.backlog_full", 1, 1.)
			atomic.AddInt64(&c.stats.AsyncErrors, 1)
			c.log("spade backlog full and cannot queue more events")
		}
	}
}

// BlockingSendEvents sends inline, not using the queue
func (c *Client) BlockingSendEvents(ctx context.Context, events ...Event) error {
	req, err := http.NewRequest("POST", c.Config.GetSpadeHost()+"/track", nil)
	if err != nil {
		return errors.Wrap(err, "cannot make request")
	}
	req.Header.Add("User-Agent", "hygienic-spade 1.0")
	encodedEvents, err := encodeBatchOfEvents(events...)
	if err != nil {
		return errors.Wrap(err, "cannot encode spade events")
	}
	vals := req.URL.Query()
	vals.Add("data", encodedEvents)
	req.URL.RawQuery = vals.Encode()

	resp, err := c.HTTPClient.Do(req)
	if err != nil {
		return errors.Wrap(err, "cannot execute spade HTTP request")
	}
	defer func() {
		if e := resp.Body.Close(); e != nil {
			c.log("err", err, "failed to close response body")
		}
	}()
	var bodyBuff bytes.Buffer
	if _, err := io.CopyN(&bodyBuff, resp.Body, 1024); err != nil && err != io.EOF {
		return errors.Wrap(err, "cannot read response body")
	}
	if _, err := io.Copy(ioutil.Discard, resp.Body); err != nil && err != io.EOF {
		return errors.Wrap(err, "cannot drain response body")
	}
	if resp.StatusCode != http.StatusNoContent {
		return fmt.Errorf("unexpected response status %d (%s)", resp.StatusCode, bodyBuff.String())
	}
	c.incStat("events", int64(len(events)), 0.1)
	atomic.AddInt64(&c.stats.TotalEventsSent, int64(len(events)))
	return nil
}
