package chatlog

import (
	"context"
	"encoding/base64"
	"encoding/json"
	"errors"
	"fmt"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/client"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/aws/aws-sdk-go/service/dynamodb/expression"

	"code.justin.tv/safety/datastore/models"
)

var (
	//ErrInvalidConfiguration indiciates the configuration is invalid
	ErrInvalidConfiguration = fmt.Errorf("The configuration is invalid")
)

// Datastore defines a list of APIs to access chatlog
type Datastore struct {
	chatDB dynamodbiface.DynamoDBAPI
}

// Config contains configuration parameters required to create a Datastore
type Config struct {
	AwsConfigProvider client.ConfigProvider
}

// Validate returns an error if the configuration is invalid
func (c *Config) Validate() error {
	if c.AwsConfigProvider == nil {
		return ErrInvalidConfiguration
	}
	return nil
}

// New returns a new chat datastore to access chatlog data
func New(conf *Config) (*Datastore, error) {
	err := conf.Validate()
	if err != nil {
		return nil, err
	}

	return &Datastore{
		chatDB: dynamodb.New(conf.AwsConfigProvider),
	}, nil
}

// ChatEntries returns a list of chat entries for the givne chat ids
func (d *Datastore) ChatEntries(ctx context.Context, ids []interface{}) ([]*models.ChatEntry, error) {
	var inputArray []map[string]*dynamodb.AttributeValue
	for _, rawID := range ids {
		id, ok := rawID.(string)
		if !ok {
			return nil, errors.New("IDs must be strings")
		}
		inputArray = append(inputArray, map[string]*dynamodb.AttributeValue{
			"MessageID": {
				S: aws.String(id),
			},
		})
	}

	input := &dynamodb.BatchGetItemInput{
		RequestItems: map[string]*dynamodb.KeysAndAttributes{
			tableChatlog: &dynamodb.KeysAndAttributes{
				Keys:           inputArray,
				ConsistentRead: aws.Bool(false),
			},
		},
	}
	result, err := d.chatDB.BatchGetItemWithContext(ctx, input)
	if err != nil {
		return nil, err
	}

	var items []*models.ChatEntry
	err = dynamodbattribute.UnmarshalListOfMaps(result.Responses[tableChatlog], &items)
	if err != nil {
		return nil, err
	}

	return items, nil
}

// FilteredChatEntriesPage returns a page of chat entries (page size is limit) that match the filters
func (d *Datastore) FilteredChatEntriesPage(ctx context.Context, user string, limit int64, cursor *string, index string, timeWindow models.TimeWindow, messageFilter *string, channelFilter *string) ([]*models.ChatEntry, *models.CursorPageInfo, error) {
	results := []*models.ChatEntry{}
	resultCount := int64(0)
	pageInfo := &models.CursorPageInfo{
		HasNext: false,
	}

	if limit < 1 {
		return nil, nil, errors.New("invalid limit, must be larger than 0")
	}

	startCursor, err := d.parseDynamoCursor(cursor)
	if err != nil {
		return nil, nil, err
	}
	if startCursor != nil {
		pageInfo.HasPrevious = true
	}

	expr, err := d.chatFilterExpression(user, index, timeWindow, channelFilter, messageFilter)
	if err != nil {
		return nil, nil, err
	}

	input := &dynamodb.QueryInput{
		ScanIndexForward:          aws.Bool(false),
		ConsistentRead:            aws.Bool(false),
		Limit:                     aws.Int64(limit),
		IndexName:                 aws.String(index),
		TableName:                 aws.String(tableChatlog),
		KeyConditionExpression:    expr.KeyCondition(),
		ExpressionAttributeNames:  expr.Names(),
		ExpressionAttributeValues: expr.Values(),
		ExclusiveStartKey:         startCursor,
	}

	filter := expr.Filter()
	if filter != nil && *filter != "" {
		input.SetFilterExpression(*filter)
	}

	var queryError error
	err = d.chatDB.QueryPagesWithContext(ctx, input, func(output *dynamodb.QueryOutput, lastPage bool) bool {
		var items []*models.ChatEntry
		queryError = dynamodbattribute.UnmarshalListOfMaps(output.Items, &items)
		if queryError != nil {
			return false
		}

		limitReached := false
		for i, item := range items {
			results = append(results, item)
			resultCount++
			if resultCount == limit {
				limitReached = true
				output = output.SetLastEvaluatedKey(d.chatKey(index, output.Items[i]))
				break
			}
		}

		pageInfo.HasNext = !lastPage
		pageInfo.EndCursor, queryError = d.encodeDynamoCursor(output.LastEvaluatedKey)
		if queryError != nil {
			return false
		}
		return !limitReached
	})
	if err != nil {
		return nil, nil, err
	}
	if queryError != nil {
		return nil, nil, queryError
	}

	return results, pageInfo, nil
}

func (d *Datastore) encodeDynamoCursor(key map[string]*dynamodb.AttributeValue) (*string, error) {
	if key == nil {
		return nil, nil
	}

	jsonCursor, err := json.Marshal(key)
	if err != nil {
		return nil, err
	}

	c := base64.StdEncoding.EncodeToString(jsonCursor)
	return &c, nil
}

func (d *Datastore) parseDynamoCursor(cursor *string) (map[string]*dynamodb.AttributeValue, error) {
	if cursor == nil || *cursor == "" {
		return nil, nil
	}

	jsonCursor, err := base64.StdEncoding.DecodeString(*cursor)
	if err != nil {
		return nil, err
	}

	var c map[string]*dynamodb.AttributeValue
	err = json.Unmarshal(jsonCursor, &c)
	if err != nil {
		return nil, err
	}

	return c, nil
}

func (d *Datastore) chatFilterExpression(user, index string, timeWindow models.TimeWindow, channelFilter *string, messageFilter *string) (expression.Expression, error) {
	const ts = "Timestamp"
	timeKey := expression.Key(ts).Between(expression.Value(timeWindow.Start), expression.Value(timeWindow.End))
	primaryKey := expression.Key(indexDynamoKeys[index]).Equal(expression.Value(user)).And(timeKey)

	builder := expression.NewBuilder().WithKeyCondition(primaryKey)

	filters := []expression.ConditionBuilder{}
	if channelFilter != nil && *channelFilter != "" {
		filters = append(filters, expression.Name(chatChannelIDKey).Equal(expression.Value(*channelFilter)))
	}
	if messageFilter != nil && *messageFilter != "" {
		filters = append(filters, expression.Or(expression.Name(chatMessageTextKey).Contains(*messageFilter),
			expression.Name(chatSystemMessageKey).Contains(*messageFilter)))
	}

	if len(filters) == 2 {
		builder = builder.WithFilter(expression.And(filters[0], filters[1]))
	} else if len(filters) == 1 {
		builder = builder.WithFilter(filters[0])
	}

	return builder.Build()
}

func (d *Datastore) chatKey(index string, value map[string]*dynamodb.AttributeValue) map[string]*dynamodb.AttributeValue {
	m := map[string]*dynamodb.AttributeValue{}
	keys := []string{chatPrimaryKey, chatGSISortKey, indexDynamoKeys[index]}
	for _, k := range keys {
		m[k] = value[k]
	}
	return m
}
