package chatlog

import (
	"context"
	"sort"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/expression"
	"github.com/pkg/errors"

	"code.justin.tv/safety/datastore/models"
)

// ChatActivity returns a user's last message in all channels the user chatted in, up to the provided limit, sorted with most recent message first
func (d *Datastore) ChatActivity(ctx context.Context, userID string, limit int) ([]*models.ChatEntry, error) {
	if limit < 0 {
		return nil, errors.New("chat activity limit cannot be less than 0")
	}

	var allItems []*models.ChatEntry

	hashCond := expression.Key(userIDKey).Equal(expression.Value(userID))
	expr, err := expression.NewBuilder().WithKeyCondition(hashCond).Build()
	if err != nil {
		return nil, err
	}

	input := &dynamodb.QueryInput{
		TableName:                 aws.String(tableChatActivityLog),
		KeyConditionExpression:    expr.KeyCondition(),
		ExpressionAttributeNames:  expr.Names(),
		ExpressionAttributeValues: expr.Values(),
		ConsistentRead:            aws.Bool(false),
	}

	var unmarshallErr error
	err = d.chatDB.QueryPagesWithContext(ctx, input, func(output *dynamodb.QueryOutput, lastPage bool) bool {
		var items []*models.ChatEntry

		err = dynamodbattribute.UnmarshalListOfMaps(output.Items, &items)
		if err != nil {
			unmarshallErr = errors.Wrap(err, "chat activity unmarshal error")
			return false
		}

		allItems = append(allItems, items...)

		return !lastPage
	})
	if unmarshallErr != nil {
		return nil, unmarshallErr
	}
	if err != nil {
		return nil, errors.Wrap(err, "chat activity query error")
	}

	result := sortEntries(allItems)
	if len(result) > limit {
		result = result[:limit]
	}

	return result, nil
}

// sortEntries sorts chat entries in descending order of Timestmap
func sortEntries(results []*models.ChatEntry) []*models.ChatEntry {
	sort.Slice(results, func(i, j int) bool {
		return results[j].Timestamp < results[i].Timestamp
	})
	return results
}
