package interfaces

import (
	"context"
	"time"

	"code.justin.tv/safety/datastore/models"
	"code.justin.tv/safety/datastore/models/shift"
)

// Datastore defines APIs to access Datastore
type Datastore interface {
	Chatlog
	Leviathan
	ReportRouter
}

// Leviathan defines a list of apis to access leviathan
type Leviathan interface {
	// Begin begins a transaction
	Begin() (Transaction, error)

	// Close closes all open connections
	Close() error
}

// Transaction defines transactions to access database
type Transaction interface {
	// Write APIs
	CreateAutoResolve(ctx context.Context, createdBy int64, resolveUntil time.Time, content, reason *string, fromUser, targetUser *models.AutoResolveUser) (*int64, error)
	CreateFileUpload(ctx context.Context, fileUpload models.FileUpload) (*int64, error)
	CreateFileUploadAudit(ctx context.Context, audit models.FileUploadAudit) (*int64, error)
	CreatePartnerEscalationOverrides(ctx context.Context, userIDs []string, override *bool) (int64, error)
	CreateReport(ctx context.Context, report models.Report) (*int64, error)
	CreateReportAudit(ctx context.Context, audit models.ReportAudit) (*int64, error)
	CreateReportComment(ctx context.Context, comment models.ReportComment) (*int64, error)
	CreateReportHold(ctx context.Context, hold models.ReportHold) (*int64, error)
	CreateSuspension(ctx context.Context, suspension models.Suspension) (*int64, error)
	CreateWarning(ctx context.Context, warning models.Warning) (*int64, error)
	CreateEnforcement(ctx context.Context, enforcement models.Enforcement) (*int64, error)
	CreateUserContentAudit(ctx context.Context, audit models.UserContentAudit) (*int64, error)
	CreateUserContentAudits(ctx context.Context, audits []*models.UserContentAudit) error
	DeletePartnerEscalationEntries(ctx context.Context, userIDs []string) (int64, error)
	ReplacePartnerEscalationList(ctx context.Context, userIDs []string) (*models.ReplacementInfo, error)
	UpdateAutoResolve(ctx context.Context, autoResolve models.AutoResolve) error
	UpdateEnforcement(ctx context.Context, enforcement models.Enforcement) error
	UpdateReport(ctx context.Context, report models.Report) error
	UpdateReportHold(ctx context.Context, hold models.ReportHold) error
	UpdateWarning(ctx context.Context, warning models.Warning) error

	// Read APIs
	ActiveReportHolds(ctx context.Context, reportIDs []int64) ([]*models.ReportHold, error)
	AdminByEmail(ctx context.Context, email string) (*models.Admin, error)
	AdminByLdap(ctx context.Context, ldap string) (*models.Admin, error)
	Admins(ctx context.Context, ids []int64) ([]*models.Admin, error)
	AdminsByLdap(ctx context.Context, ldaps []string) ([]*models.Admin, error)
	AutoResolve(ctx context.Context, id int64) (*models.AutoResolve, error)
	AutoResolves(ctx context.Context, ids []int64) ([]*models.AutoResolve, error)
	BanExceptionsByUsers(ctx context.Context, userIDs []int64) ([]*models.BanException, error)
	EscalationPartnersByID(ctx context.Context, userIDs []string) ([]*models.EscalationPartner, error)
	EscalationPartnersPage(ctx context.Context, limit uint64, offset uint64) ([]*models.EscalationPartner, error)
	EscalationPartnersCount(ctx context.Context) (int64, error)
	FindActiveAutoResolves(ctx context.Context, content, reason string, fromUserID, targetUserID *int) ([]*models.AutoResolve, error)
	FindAllActiveAutoResolves(ctx context.Context, limit, offset uint64) ([]*models.AutoResolve, *models.PageInfo, error)
	LatestSuspensionForUsers(ctx context.Context, userIDs []int64) ([]*models.Suspension, error)
	Report(ctx context.Context, id int64) (*models.Report, error)
	ReportAudit(ctx context.Context, id int64) (*models.ReportAudit, error)
	ReportAuditsByReport(ctx context.Context, reportID int64) ([]*models.ReportAudit, error)
	ReportAuditsByReports(ctx context.Context, reportIDs []int64) ([]*models.ReportAudit, error)
	ReportComments(ctx context.Context, ids []int64) ([]*models.ReportComment, error)
	ReportCommentsByReport(ctx context.Context, reportID int64, limit, offset uint64) ([]*models.ReportComment, *models.PageInfo, error)
	ReportCount(ctx context.Context, userID int64, start time.Time, end time.Time) (int64, error)
	ReportCountByFromUser(ctx context.Context, userID int64, start time.Time, end time.Time) (int64, error)
	ReportHolds(ctx context.Context, ids []int64) ([]*models.ReportHold, error)
	ReportHoldsByAdmin(ctx context.Context, adminID *int64, limit uint64, cursor *int64) ([]*models.ReportHold, *models.PageInfo, error)
	ReportPage(ctx context.Context, filter *models.ReportFilter, sort *models.ReportSort, limit uint64, offset uint64) ([]*models.Report, *models.PageInfo, error)
	Reports(ctx context.Context, ids []int64) ([]*models.Report, error)
	Suspension(ctx context.Context, id int64) (*models.Suspension, error)
	Enforcement(ctx context.Context, id int64) (*models.Enforcement, error)
	Warning(ctx context.Context, id int64) (*models.Warning, error)
	SuspensionGuideContents(ctx context.Context, codes []string) ([]*models.SuspensionGuideContent, error)
	SuspensionGuideContentsAll(ctx context.Context) ([]*models.SuspensionGuideContent, error)
	SuspensionGuideDetailedEntries(ctx context.Context, content, reason string) ([]*models.SuspensionGuideDetailedEntry, error)
	SuspensionGuideDetailedReasonsByCode(ctx context.Context, detailedReasons []string) ([]*models.SuspensionGuideDetailedReason, error)
	SuspensionGuideDetailedReasonsAll(ctx context.Context) ([]*models.SuspensionGuideDetailedReason, error)
	SuspensionGuideQuickSelects(ctx context.Context) ([]*models.SuspensionGuideDetailedEntry, error)
	SuspensionGuideReasons(ctx context.Context, codes []string) ([]*models.SuspensionGuideReason, error)
	SuspensionGuideReasonsAll(ctx context.Context) ([]*models.SuspensionGuideReason, error)
	SuspensionGuideReasonsByContent(ctx context.Context, content string) ([]*models.SuspensionGuideReason, error)
	SuspensionGuides(ctx context.Context) ([]*models.SuspensionGuide, error)
	SuspensionPage(ctx context.Context, filter *models.SuspensionFilter, sort *models.SuspensionSort, limit uint64, offset uint64) ([]*models.Suspension, *models.PageInfo, error)
	WarningPage(ctx context.Context, filter *models.WarningFilter, sort *models.WarningSort, limit uint64, offset uint64) ([]*models.Warning, *models.PageInfo, error)
	EnforcementPage(ctx context.Context, filter *models.EnforcementFilter, sort *models.EnforcementSort, limit uint64, offset uint64) ([]*models.Enforcement, *models.PageInfo, error)
	Suspensions(ctx context.Context, ids []int64) ([]*models.Suspension, error)
	Enforcements(ctx context.Context, ids []int64) ([]*models.Enforcement, error)
	Warnings(ctx context.Context, ids []int64) ([]*models.Warning, error)
	UserContentAuditByUser(ctx context.Context, userID string, limit, offset uint64) ([]*models.UserContentAudit, *models.PageInfo, error)
	UserContentAudits(ctx context.Context, ids []int64) ([]*models.UserContentAudit, error)

	// Commit commits the transaction
	Commit() error
	// Rollback rolls back the transaction
	Rollback() error
	// SafeRollback rolls back the transaction if only it hadn't been committed/already rolled back.
	// This also swallows any errors returned from rollback. use this for `defer tx.SafeRollback()`
	SafeRollback(ctx context.Context)
}

// Chatlog defines APIs for accessing chatlog data
type Chatlog interface {
	ChatEntries(ctx context.Context, ids []interface{}) ([]*models.ChatEntry, error)
	FilteredChatEntriesPage(ctx context.Context, user string, limit int64, cursor *string, index string, timeWindow models.TimeWindow, messageFilter *string, channelFilter *string) ([]*models.ChatEntry, *models.CursorPageInfo, error)
	ChatActivity(ctx context.Context, userID string, limit int) ([]*models.ChatEntry, error)
}

// ReportRouter defines APIs for accessing shift and report routing data
type ReportRouter interface {
	// Write APIs
	CreateShift(adminID int64, shiftType shift.Type, weight int64) (*models.Shift, error)
	CreateRoute(shiftID string, targetUserID string) error
	RefreshRoute(targetUserID string) error
	DeleteShift(shiftID string) error
	DeleteRoute(targetUserID string) error
	DeleteRoutes(shiftID string) error
	DeleteAllRoutes() error
	AddReportToShift(shiftID string, reportID int64) error
	RemoveReportFromShift(shiftID string, reportID int64) error
	RemoveAllReportsFromShift(shiftID string) error

	// Read APIs
	Shift(shiftID string) (*models.Shift, error)
	Shifts() ([]models.Shift, error)
	ShiftsByAdmin(adminID int64) ([]models.Shift, error)
	RoutedReports(shiftID string) ([]int64, error)
	Route(targetUserID string) (*models.Shift, error)

	// Util
	ShiftID(adminID int64, shiftType shift.Type) string
	IsValidType(shiftType shift.Type) bool
}
