package leviathan

import (
	"context"
	"fmt"

	"github.com/Masterminds/squirrel"
	"github.com/pkg/errors"

	"code.justin.tv/safety/datastore/models"
)

// Admins returns the list of admins for given ids
func (t *Transaction) Admins(ctx context.Context, ids []int64) ([]*models.Admin, error) {
	q := squirrel.Select("*").From(tableUsers).
		Where(squirrel.Eq{"id": ids})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var result []*models.Admin
	err = t.tx.SelectContext(ctx, &result, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return result, nil
}

// AdminsByLdap returns the list of admins for given ldaps
func (t *Transaction) AdminsByLdap(ctx context.Context, ldaps []string) ([]*models.Admin, error) {
	q := squirrel.Select("*").From(tableUsers).
		Where(squirrel.Eq{"ldap_id": ldaps})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var result []*models.Admin
	err = t.tx.SelectContext(ctx, &result, sql, args...)
	return result, errors.Wrap(err, msgSelectContext)
}

// AdminByEmail returns the admin by email address
func (t *Transaction) AdminByEmail(ctx context.Context, email string) (*models.Admin, error) {
	q := squirrel.Select("*").From(tableUsers).
		Where(squirrel.Eq{"email": email})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var result []*models.Admin
	err = t.tx.SelectContext(ctx, &result, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	if len(result) == 0 {
		return nil, nil
	}

	if len(result) != 1 {
		return nil, fmt.Errorf("Found more than 1 admins by email %s", email)
	}

	return result[0], nil
}

// AdminByLdap returns the admin by ldap
func (t *Transaction) AdminByLdap(ctx context.Context, ldap string) (*models.Admin, error) {
	q := squirrel.Select("*").From(tableUsers).
		Where(squirrel.Eq{"ldap_id": ldap})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var result []*models.Admin
	err = t.tx.SelectContext(ctx, &result, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	if len(result) == 0 {
		return nil, nil
	}

	if len(result) != 1 {
		return nil, fmt.Errorf("Found more than 1 admins by ldap %s", ldap)
	}

	return result[0], nil
}
