package leviathan

import (
	"context"
	"time"

	"code.justin.tv/safety/datastore/models"

	"github.com/Masterminds/squirrel"
	"github.com/pkg/errors"
)

// FindActiveAutoResolves returns all active auto resolve record that matches query, or an empty array if none matches
func (t *Transaction) FindActiveAutoResolves(ctx context.Context, content, reason string, fromUserID, targetUserID *int) ([]*models.AutoResolve, error) {
	// Negative user id represents "NOT"
	sb := squirrel.Select("*").From(tableAutoResolves)
	sb = onlyActiveAutoresolves(sb)
	q := sb.
		// If the rule does not specify a content, treat that as a wildcard match for all report contents
		// If an auto resolve rule specified a content, that content must match the content of the report
		Where(squirrel.Or{
			squirrel.Eq{"content": nil},
			squirrel.Eq{"content": content},
		}).
		// If the rule does not specify a reason, treat that as a wildcard match for all report reasons
		// If an auto resolve rule specified a reason, that reason must match the reason of the report
		Where(squirrel.Or{
			squirrel.Eq{"reason": nil},
			squirrel.Eq{"reason": reason},
		}).
		// Rule matches all from user id or the specified user,
		// or user is not specified user
		// The NOT operation is specified by negative ID of the user. If user id is
		// negative, then would want to only match if report has a different ID
		// TODO replace negative id with something more sensible by SAFETY-1523
		Where(squirrel.Or{
			squirrel.Eq{"from_user_id": nil},
			squirrel.Eq{"from_user_id": fromUserID},
			squirrel.And{
				squirrel.Lt{"from_user_id": 0},
				squirrel.NotEq{"ABS(from_user_id)": fromUserID},
			},
		}).
		// Rule matches all target user id or the specified user,
		// or user is not specified user
		// The NOT operation is specified by negative ID of the user. If user id is
		// negative, then would want to only match if report has a different ID
		// TODO replace negative id with something more sensible by SAFETY-1523
		Where(squirrel.Or{
			squirrel.Eq{"target_user_id": nil},
			squirrel.Eq{"target_user_id": targetUserID},
			squirrel.And{
				squirrel.Lt{"target_user_id": 0},
				squirrel.NotEq{"ABS(target_user_id)": targetUserID},
			},
		})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var result []*models.AutoResolve
	err = t.tx.SelectContext(ctx, &result, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return result, nil
}

// UpdateAutoResolve updates an autoresolve record
func (t *Transaction) UpdateAutoResolve(ctx context.Context, autoResolve models.AutoResolve) error {
	autoResolve.UpdatedAt = time.Now()

	sql, args, err := toUpdateStatement(tableAutoResolves, autoResolve)
	if err != nil {
		return errors.Wrap(err, msgSQLConversion)
	}

	_, err = t.tx.ExecContext(ctx, sql, args...)
	if err != nil {
		return errors.Wrap(err, msgUpdateContext)
	}

	return nil
}

// FindAllActiveAutoResolves returns all active auto resolves
func (t *Transaction) FindAllActiveAutoResolves(ctx context.Context, limit, offset uint64) ([]*models.AutoResolve, *models.PageInfo, error) {
	q := onlyActiveAutoresolves(squirrel.Select("*").From(tableAutoResolves)).Limit(limit).Offset(offset)

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSQLConversion)
	}

	var result []*models.AutoResolve
	err = t.tx.SelectContext(ctx, &result, sql, args...)
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSelectContext)
	}

	q = onlyActiveAutoresolves(squirrel.Select("count(*) as total").From(tableAutoResolves))

	sql, args, err = q.ToSql()
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSQLConversion)
	}

	var pageInfo []*models.PageInfo
	err = t.tx.SelectContext(ctx, &pageInfo, sql, args...)
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSelectContext)
	}

	if len(pageInfo) != 1 {
		return nil, nil, errors.New("Incorrect number of results returned for page info")
	}

	return result, pageInfo[0], nil
}

// CreateAutoResolve creates an auto resolve rule
func (t *Transaction) CreateAutoResolve(ctx context.Context, createdBy int64, resolveUntil time.Time, content, reason *string, fromUser, targetUser *models.AutoResolveUser) (*int64, error) {
	if content == nil && reason == nil {
		return nil, errors.New("One of content or reason must be specified")
	}

	if fromUser == nil && targetUser == nil {
		return nil, errors.New("One of target user or from user must be specified")
	}

	// Both target and from user cannot be a range operation (nil or IS NOT).
	// Allowing both to be range operator will likely result in unintended report getting resolved
	if targetUser == nil || targetUser.OperationType == models.AutoResolveUserIsNot {
		if fromUser == nil || fromUser.OperationType == models.AutoResolveUserIsNot {
			return nil, errors.New("From user must be IS if target user is not specified")
		}
	}

	autoResolve := models.AutoResolve{
		CreatedBy:    &createdBy,
		CreatedAt:    time.Now(),
		Content:      content,
		FromUserID:   autoResolveUserToUserID(fromUser),
		Reason:       reason,
		ResolveUtil:  &resolveUntil,
		TargetUserID: autoResolveUserToUserID(targetUser),
		UpdatedAt:    time.Now(),
	}

	sql, args, err := toInsertStatement(tableAutoResolves, autoResolve)
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	sqlResult, err := t.tx.ExecContext(ctx, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgInsertContext)
	}

	id, err := sqlResult.LastInsertId()
	if err != nil {
		return nil, errors.Wrap(err, msgRetrieveLastID)
	}

	return &id, nil
}

// AutoResolve returns the auto resolve of the given id
func (t *Transaction) AutoResolve(ctx context.Context, id int64) (*models.AutoResolve, error) {
	autoResolves, err := t.AutoResolves(ctx, []int64{id})
	if err != nil {
		return nil, err
	}

	if len(autoResolves) == 0 {
		return nil, nil
	}

	if len(autoResolves) == 1 {
		return autoResolves[0], nil
	}

	return nil, errors.Errorf("Found more than one auto resolve of id %d", id)
}

// AutoResolves returns autoResolves with given ids
func (t *Transaction) AutoResolves(ctx context.Context, ids []int64) ([]*models.AutoResolve, error) {
	q := squirrel.Select("*").From(tableAutoResolves).
		Where(squirrel.Eq{"id": ids})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var autoResolves []*models.AutoResolve
	err = t.tx.SelectContext(ctx, &autoResolves, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return autoResolves, nil
}

// TODO replace negative id with something more sensible by SAFETY-1523
func autoResolveUserToUserID(u *models.AutoResolveUser) *int {
	if u == nil {
		return nil
	}
	if u.OperationType == models.AutoResolveUserIs {
		return &u.UserID
	}
	id := u.UserID * -1
	return &id
}

func onlyActiveAutoresolves(builder squirrel.SelectBuilder) squirrel.SelectBuilder {
	return builder.Where(squirrel.Eq{"disabled_at": nil}).
		Where(squirrel.Gt{"resolve_until": time.Now()})
}
