package leviathan

import (
	"context"
	"fmt"
	"time"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/safety/datastore/interfaces"
	"code.justin.tv/safety/datastore/models"

	"github.com/go-sql-driver/mysql"
	"github.com/jmoiron/sqlx"
	"github.com/pkg/errors"
)

var (
	//ErrInvalidConfiguration indiciates the configuration is invalid
	ErrInvalidConfiguration = fmt.Errorf("The configuration is invalid")
)

const (
	defaultMaxIdleConn = 5
)

// Datastore defines a list of APIs to access Aegis data
type Datastore struct {
	db sqlxDB
}

// Transaction defines a llist of APIs to access data in a transaction
// Commit must be Rollback/SafeRollback must be
type Transaction struct {
	isActive bool
	tx       sqlxTx
}

// Config contains configuration parameters required to create a Datastore
type Config struct {
	User         string
	Password     string `json:"-"`
	Address      string
	DatabaseName string
	Timeout      time.Duration // Dial timeout
	ReadTimeout  time.Duration // I/O read timeout
	WriteTimeout time.Duration // I/O write timeout

	// Maximum idle connections to keep alive, default to 5.
	// Current (Mar 14, 2019) max conn is 648.
	// Sum of this value for all services that talks to Leviathan
	// cannot be higher than 648 (NEED TO leave some room to for bursty connections)
	// Non definite list
	// - Aegis 16 * 3 = 48
	// - Gateway 5 * 3 = 15
	// - Distinguish 5 * 3 = 15
	// - Levithan 3 * 1 = 3
	MaxIdleConnections int
}

// Validate returns an error if the configuration is invalid
func (c *Config) Validate() error {
	if c.User == "" || c.Password == "" || c.Address == "" || c.DatabaseName == "" {
		return ErrInvalidConfiguration
	}
	return nil
}

// New creates a new datastore
func New(conf *Config) (*Datastore, error) {
	err := conf.Validate()
	if err != nil {
		return nil, err
	}

	cfg := mysql.NewConfig()

	cfg.User = conf.User
	cfg.Passwd = conf.Password
	cfg.Net = "tcp"
	cfg.Addr = conf.Address
	cfg.DBName = conf.DatabaseName
	cfg.ParseTime = true
	cfg.Timeout = conf.Timeout
	cfg.ReadTimeout = conf.ReadTimeout
	cfg.WriteTimeout = conf.WriteTimeout
	cfg.Collation = "utf8mb4_general_ci"

	ctxTimeout := conf.Timeout
	if ctxTimeout == 0 {
		ctxTimeout = 1 * time.Second
	}

	ctx := context.Background()
	ctx, cancel := context.WithTimeout(ctx, ctxTimeout)
	defer cancel()

	db, err := sqlx.ConnectContext(ctx, "mysql", cfg.FormatDSN())
	if err != nil {
		return nil, err
	}

	db = db.Unsafe()
	maxIdleConn := defaultMaxIdleConn
	if conf.MaxIdleConnections > 0 {
		maxIdleConn = conf.MaxIdleConnections
	}
	db.SetMaxIdleConns(maxIdleConn)

	// Avoid invalid connections after too long idles
	db.SetConnMaxLifetime(time.Hour)

	return &Datastore{db: db}, nil
}

// Begin begins a new transaction
func (d *Datastore) Begin() (interfaces.Transaction, error) {
	tx, err := d.db.Beginx()
	if err != nil {
		return nil, err
	}

	return &Transaction{
		tx:       tx,
		isActive: true,
	}, nil
}

// Commit commits the transaction
func (t *Transaction) Commit() error {
	if t.isActive {
		return t.updateComplete(t.tx.Commit())
	}
	return models.ErrNoActiveTransaction
}

// Rollback rolls back the transaction
func (t *Transaction) Rollback() error {
	if t.isActive {
		return t.updateComplete(t.tx.Rollback())
	}
	return models.ErrNoActiveTransaction
}

// SafeRollback rolls back the transaction and swallow any error occurred during the rollback
func (t *Transaction) SafeRollback(ctx context.Context) {
	if !t.isActive {
		return
	}
	// Always set it to inactive regardless of the result.
	// Nothing is expected to be done after this function is invoked
	t.isActive = false

	err := t.tx.Rollback()
	if err != nil {
		logx.Error(ctx, errors.Wrap(err, "Unable to rollback"))
	}
}

// Updates the transaction based on error. handy method to check commit/rollbacks
// Only mark the transaction as completed if there's no error executing previous command
func (t *Transaction) updateComplete(err error) error {
	if err == nil {
		t.isActive = false
	}
	return err
}

// Close closes all open connections to the datastore
func (d *Datastore) Close() error {
	return d.db.Close()
}
