package leviathan

import (
	"context"
	"time"

	"github.com/Masterminds/squirrel"
	"github.com/pkg/errors"

	"code.justin.tv/safety/datastore/models"
)

// A simple structure that holds count for SQL result auto fill
type count struct {
	Count int64 `db:"count"`
}

// CreateReport creates a new report record in database
func (t *Transaction) CreateReport(ctx context.Context, report models.Report) (*int64, error) {
	now := time.Now()
	report.CreatedAt = now
	report.UpdatedAt = now

	sql, args, err := toInsertStatement(tableReports, report)
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	sqlResult, err := t.tx.ExecContext(ctx, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgInsertContext)
	}

	// Update ID after we successfully insert report
	id, err := sqlResult.LastInsertId()
	if err != nil {
		return nil, errors.Wrap(err, msgRetrieveLastID)
	}
	return &id, nil
}

// UpdateReport updates a report record in database
func (t *Transaction) UpdateReport(ctx context.Context, report models.Report) error {
	report.UpdatedAt = time.Now()

	sql, args, err := toUpdateStatement(tableReports, report)
	if err != nil {
		return errors.Wrap(err, msgSQLConversion)
	}

	_, err = t.tx.ExecContext(ctx, sql, args...)
	if err != nil {
		return errors.Wrap(err, msgUpdateContext)
	}

	return nil
}

// Report returns report for given id
func (t *Transaction) Report(ctx context.Context, id int64) (*models.Report, error) {
	reports, err := t.Reports(ctx, []int64{id})
	if err != nil {
		return nil, err
	}

	if len(reports) == 0 {
		return nil, nil
	}

	if len(reports) == 1 {
		return reports[0], nil
	}

	return nil, errors.Errorf("Found more than one report of id %d", id)
}

// Reports returns reports for given ids
func (t *Transaction) Reports(ctx context.Context, ids []int64) ([]*models.Report, error) {
	q := squirrel.Select("*").From(tableReports).
		Where(squirrel.Eq{"id": ids})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var reports []*models.Report
	err = t.tx.SelectContext(ctx, &reports, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return reports, nil
}

// ReportPage returns reports up to limit, resuming from offset (use 0 for first call) while keeping filters and sorts
func (t *Transaction) ReportPage(ctx context.Context, filter *models.ReportFilter, sort *models.ReportSort, limit uint64, offset uint64) ([]*models.Report, *models.PageInfo, error) {
	var err error
	q := squirrel.Select("*").From(tableReports).Limit(limit).Offset(offset)

	q, err = sortBy(q, sort)
	if err != nil {
		return nil, nil, err
	}

	q, err = filterBy(q, filter)
	if err != nil {
		return nil, nil, err
	}

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSQLConversion)
	}

	var reports []*models.Report
	err = t.tx.SelectContext(ctx, &reports, sql, args...)
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSelectContext)
	}

	q = squirrel.Select("count(*) as total").From(tableReports)
	q, err = filterBy(q, filter)
	if err != nil {
		return nil, nil, err
	}
	sql, args, err = q.ToSql()
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSQLConversion)
	}

	var pageInfo []*models.PageInfo
	err = t.tx.SelectContext(ctx, &pageInfo, sql, args...)
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSelectContext)
	}

	if len(pageInfo) != 1 {
		return nil, nil, errors.New("Incorrect number of results returned for page info")
	}

	return reports, pageInfo[0], nil
}

// ReportCount returns number of reports against a given user in the specified time range
func (t *Transaction) ReportCount(ctx context.Context, userID int64, start time.Time, end time.Time) (int64, error) {
	if start.After(end) {
		return 0, errors.Errorf("Start (%v) cannot be after end time (%v)", start, end)
	}
	q := squirrel.Select("count(*) as count").
		From(tableReports).
		Where(squirrel.Eq{"target_user_id": userID}).
		Where(squirrel.Gt{"created_at": start}).
		Where(squirrel.Lt{"created_at": end})

	sql, args, err := q.ToSql()
	if err != nil {
		return 0, errors.Wrap(err, msgSQLConversion)
	}

	var counts []*count
	err = t.tx.SelectContext(ctx, &counts, sql, args...)
	if err != nil {
		return 0, errors.Wrap(err, msgSelectContext)
	}

	if len(counts) != 1 {
		return 0, errors.New("Incorrect number of result returned for counts")
	}

	return counts[0].Count, nil
}

// ReportCountByFromUser returns number of reports submitted a given user in the specified time range
func (t *Transaction) ReportCountByFromUser(ctx context.Context, userID int64, start time.Time, end time.Time) (int64, error) {
	if start.After(end) {
		return 0, errors.Errorf("Start (%v) cannot be after end time (%v)", start, end)
	}

	q := squirrel.Select("count(*) as count").
		From(tableReports).
		Where(squirrel.Eq{"from_user_id": userID}).
		Where(squirrel.Gt{"created_at": start}).
		Where(squirrel.Lt{"created_at": end})

	sql, args, err := q.ToSql()
	if err != nil {
		return 0, errors.Wrap(err, msgSQLConversion)
	}

	var counts []*count
	err = t.tx.SelectContext(ctx, &counts, sql, args...)
	if err != nil {
		return 0, errors.Wrap(err, msgSelectContext)
	}

	if len(counts) != 1 {
		return 0, errors.New("Incorrect number of result returned for counts")
	}

	return counts[0].Count, nil
}
