package leviathan

import (
	"context"
	"time"

	"code.justin.tv/safety/datastore/models"
	"github.com/Masterminds/squirrel"
	"github.com/pkg/errors"
)

// CreateReportAudit creates a new report record in database
func (t *Transaction) CreateReportAudit(ctx context.Context, audit models.ReportAudit) (*int64, error) {
	now := time.Now()
	audit.CreatedAt = now
	audit.UpdatedAt = now

	sql, args, err := toInsertStatement(tableReportAudits, audit)
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	sqlResult, err := t.tx.ExecContext(ctx, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgInsertContext)
	}

	// Update ID after we successfully insert report
	id, err := sqlResult.LastInsertId()
	if err != nil {
		return nil, errors.Wrap(err, msgRetrieveLastID)
	}

	return &id, nil
}

// ReportAudit returns a report audit entry for the given id
func (t *Transaction) ReportAudit(ctx context.Context, id int64) (*models.ReportAudit, error) {
	q := squirrel.Select("*").From(tableReportAudits).
		Where(squirrel.Eq{"id": id})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var reportAudits []*models.ReportAudit
	err = t.tx.SelectContext(ctx, &reportAudits, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	if len(reportAudits) == 0 {
		return nil, nil
	}

	if len(reportAudits) == 1 {
		return reportAudits[0], nil
	}

	return nil, errors.Errorf("Found more than one report audit of id %d", id)
}

// ReportAuditsByReport returns audits for a given report
func (t *Transaction) ReportAuditsByReport(ctx context.Context, reportID int64) ([]*models.ReportAudit, error) {
	q := squirrel.Select("*").From(tableReportAudits).
		Where(squirrel.Eq{"report_id": reportID}).
		OrderBy("created_at desc")

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var audits []*models.ReportAudit
	err = t.tx.SelectContext(ctx, &audits, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	if len(audits) > 0 {
		return audits, nil
	}

	return nil, nil
}

// ReportAuditsByReports returns audits for a list of reports ordered by created_at (not by report_id)
func (t *Transaction) ReportAuditsByReports(ctx context.Context, reportIDs []int64) ([]*models.ReportAudit, error) {
	q := squirrel.Select("*").From(tableReportAudits).
		Where(squirrel.Eq{"report_id": reportIDs}).
		OrderBy("created_at desc")

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var audits []*models.ReportAudit
	err = t.tx.SelectContext(ctx, &audits, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	if len(audits) > 0 {
		return audits, nil
	}

	return nil, nil
}
