package leviathan

import (
	"context"
	"time"

	"github.com/Masterminds/squirrel"
	"github.com/pkg/errors"

	"code.justin.tv/safety/datastore/models"
)

// CreateReportComment creates a new report comment
func (t *Transaction) CreateReportComment(ctx context.Context, comment models.ReportComment) (*int64, error) {
	now := time.Now()
	comment.CreatedAt = now
	comment.UpdatedAt = now

	sql, args, err := toInsertStatement(tableReportComments, comment)
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	sqlResult, err := t.tx.ExecContext(ctx, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgInsertContext)
	}

	// Update ID after we successfully insert report comment
	id, err := sqlResult.LastInsertId()
	if err != nil {
		return nil, errors.Wrap(err, msgRetrieveLastID)
	}
	return &id, nil
}

// ReportComments returns reports comments for given ids
func (t *Transaction) ReportComments(ctx context.Context, ids []int64) ([]*models.ReportComment, error) {
	q := squirrel.Select("*").From(tableReportComments).
		Where(squirrel.Eq{"id": ids})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var comments []*models.ReportComment
	err = t.tx.SelectContext(ctx, &comments, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return comments, nil
}

// ReportCommentsByReport returns all comments for a given report
func (t *Transaction) ReportCommentsByReport(ctx context.Context, reportID int64, limit, offset uint64) ([]*models.ReportComment, *models.PageInfo, error) {
	q := squirrel.Select("*").From(tableReportComments).
		Where(squirrel.Eq{"report_id": reportID}).
		Limit(limit).
		Offset(offset)

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSQLConversion)
	}

	var reportComments []*models.ReportComment
	err = t.tx.SelectContext(ctx, &reportComments, sql, args...)
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSelectContext)
	}

	q = squirrel.Select("count(*) as total").
		Where(squirrel.Eq{"report_id": reportID}).
		From(tableReportComments)

	sql, args, err = q.ToSql()
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSQLConversion)
	}

	var pageInfo []*models.PageInfo
	err = t.tx.SelectContext(ctx, &pageInfo, sql, args...)
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSelectContext)
	}

	if len(pageInfo) != 1 {
		return nil, nil, errors.New("Incorrect number of results returned for page info")
	}
	return reportComments, pageInfo[0], nil
}
