package leviathan

import (
	"context"
	"fmt"
	"time"

	"github.com/Masterminds/squirrel"
	"github.com/pkg/errors"

	"code.justin.tv/safety/datastore/models"
)

// CreateReportHold creates a new report hold record
/*
 * The structure of this query is as follows:
 *
 * INSERT INTO table
 * SELECT * FROM (
 *   SELECT ?, ?, ?
 * ) AS tmp WHERE NOT EXISTS (
 *   SELECT id FROM table WHERE filter = ?
 * )
 *
 * It basically says "return the row from the first subquery only if there are no
 * rows in the second subquery". The second subquery is the one that tells us if
 * there is a hold already in place. Then we use the result to insert.
 *
 * This is basically an "INSERT IF NOT EXISTS" but that works with arbitrary
 * filters, not just unique indexes.
 */
func (t *Transaction) CreateReportHold(ctx context.Context, hold models.ReportHold) (*int64, error) {
	now := time.Now()

	hold.CreatedAt = now
	hold.UpdatedAt = now

	filterQuery := squirrel.Select("id").From(tableReportHolds).
		Where(squirrel.Gt{"hold_until": now}).
		Where(squirrel.Eq{"report_id": hold.ReportID}).
		Where(squirrel.Eq{"disabled_at": nil})

	filterSQL, filterArgs, err := filterQuery.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	columns, insertArgs := toColumnsAndValues(hold)
	for i := range columns {
		columns[i] = "? AS " + columns[i]
	}

	q := squirrel.Insert(tableReportHolds).Select(
		squirrel.Select("*").FromSelect(squirrel.Select(columns...), "tmp").
			Where(fmt.Sprintf("NOT EXISTS (%s)", filterSQL)))

	sql, _, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	args := append(insertArgs, filterArgs...)

	sqlResult, err := t.tx.ExecContext(ctx, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgInsertContext)
	}

	numRows, err := sqlResult.RowsAffected()
	if err != nil {
		return nil, errors.Wrap(err, msgRetrieveRowsAffected)
	}

	if numRows == 0 {
		return nil, errors.Errorf("There is an active report hold already in place for report id %d", hold.ReportID)
	}

	id, err := sqlResult.LastInsertId()
	if err != nil {
		return nil, errors.Wrap(err, msgRetrieveLastID)
	}

	return &id, nil
}

// UpdateReportHold Modifies the report hold record
func (t *Transaction) UpdateReportHold(ctx context.Context, hold models.ReportHold) error {
	hold.UpdatedAt = time.Now()

	sql, args, err := toUpdateStatement(tableReportHolds, hold)
	if err != nil {
		return errors.Wrap(err, msgSQLConversion)
	}

	sqlResult, err := t.tx.ExecContext(ctx, sql, args...)
	if err != nil {
		return errors.Wrap(err, msgSelectContext)
	}

	if numRows, err := sqlResult.RowsAffected(); err != nil {
		return err
	} else if numRows != 1 {
		return errors.Errorf("The update operation for report hold affected %d rows, 1 expected.", numRows)
	}

	return nil
}

// ReportHolds retrieves the report holds record by ids.
func (t *Transaction) ReportHolds(ctx context.Context, ids []int64) ([]*models.ReportHold, error) {
	q := squirrel.Select("*").From(tableReportHolds).
		Where(squirrel.Eq{"id": ids})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var holds []*models.ReportHold
	err = t.tx.SelectContext(ctx, &holds, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	if len(holds) == 0 {
		return nil, nil
	}

	return holds, nil
}

// ActiveReportHolds returns the active hold for each one of the report ids given
func (t *Transaction) ActiveReportHolds(ctx context.Context, reportIDs []int64) ([]*models.ReportHold, error) {
	now := time.Now()

	q := squirrel.Select("*").From(tableReportHolds).
		Where(squirrel.Eq{"report_id": reportIDs}).
		Where(squirrel.Gt{"hold_until": now}).
		Where(squirrel.Eq{"disabled_at": nil})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var reportHolds []*models.ReportHold
	err = t.tx.SelectContext(ctx, &reportHolds, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	idMap := make(map[int64]bool)
	for _, hold := range reportHolds {
		if _, ok := idMap[hold.ReportID]; ok {
			return nil, errors.Errorf("Invalid state: Multiple records match report hold for report id %d", hold.ReportID)
		}
		idMap[hold.ReportID] = true
	}

	return reportHolds, nil
}

// ReportHoldsByAdmin returns the set of active report holds created by a given admin
// a nil adminID will retrieve the reports on hold for all admins
func (t *Transaction) ReportHoldsByAdmin(ctx context.Context, adminID *int64, limit uint64, cursor *int64) ([]*models.ReportHold, *models.PageInfo, error) {
	now := time.Now()

	q := squirrel.Select("*").From(tableReportHolds).
		Where(squirrel.Gt{"hold_until": now}).
		Where(squirrel.Eq{"disabled_at": nil}).
		OrderBy("id asc").
		Limit(limit)

	if adminID != nil {
		q = q.Where(squirrel.Eq{"created_by": adminID})
	}

	if cursor != nil {
		q = q.Where(squirrel.Gt{"id": &cursor})
	}

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSQLConversion)
	}

	var holds []*models.ReportHold
	err = t.tx.SelectContext(ctx, &holds, sql, args...)
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSelectContext)
	}

	q = squirrel.Select("COALESCE(min(id), 0) as first_id, COALESCE(max(id), 0) as last_id, count(1) as total").From(tableReportHolds).
		Where(squirrel.Gt{"hold_until": now}).
		Where(squirrel.Eq{"disabled_at": nil}).
		OrderBy("id asc").
		Limit(limit)

	if adminID != nil {
		q = q.Where(squirrel.Eq{"created_by": adminID})
	}

	sql, args, err = q.ToSql()
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSQLConversion)
	}

	var pageInfo []*models.PageInfo
	err = t.tx.SelectContext(ctx, &pageInfo, sql, args...)
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSelectContext)
	}

	if len(pageInfo) != 1 {
		return nil, nil, errors.New("Incorrect number of results returned for page info")
	}

	return holds, pageInfo[0], nil
}
