package leviathan

import (
	"reflect"
	"strings"

	"github.com/Masterminds/squirrel"
)

// toUpdateStatement converts the model to sql statement that will update the record
func toUpdateStatement(table string, model interface{}) (string, []interface{}, error) {
	columns, values := toColumnsAndValues(model)

	var id interface{}
	clauses := map[string]interface{}{}
	for i := 0; i < len(columns); i++ {
		switch strings.ToLower(columns[i]) {
		case columnID:
			id = values[i]
		default:
			clauses[columns[i]] = values[i]
		}
	}

	return squirrel.Update(table).SetMap(clauses).Where("id = ?", id).ToSql()
}

// toInsertStatements converts the model to sql statement that will insert the record
func toInsertStatement(table string, model interface{}) (string, []interface{}, error) {
	columns, values := toColumnsAndValues(model)
	return squirrel.Insert(table).Columns(columns...).Values(values...).ToSql()
}

// toColumnsAndValues to converts a model into columns and their respective value
func toColumnsAndValues(model interface{}) ([]string, []interface{}) {
	// Need to dereference pointers before we can inspect the values and tags
	if reflect.TypeOf(model).Kind() == reflect.Ptr {
		return toColumnsAndValues(reflect.ValueOf(model).Elem().Interface())
	}

	values := reflect.ValueOf(model)
	types := values.Type()

	sqlColumns := make([]string, 0, types.NumField())
	sqlValues := make([]interface{}, 0, types.NumField())

	for i := 0; i < types.NumField(); i++ {
		t := types.Field(i)
		v := values.Field(i)

		// Skip unexported fields
		if t.PkgPath != "" || v.Kind() == reflect.Invalid {
			continue
		}

		// Prefer tags over field name for column name
		n := t.Tag.Get("db")
		if n == "" {
			n = t.Name
		}

		sqlColumns = append(sqlColumns, n)
		sqlValues = append(sqlValues, v.Interface())
	}

	return sqlColumns, sqlValues
}
