package leviathan

import (
	"context"
	"fmt"
	"time"

	"github.com/Masterminds/squirrel"
	"github.com/pkg/errors"

	"code.justin.tv/safety/datastore/models"
)

// CreateSuspension creates a new suspension record in database
func (t *Transaction) CreateSuspension(ctx context.Context, suspension models.Suspension) (*int64, error) {
	now := time.Now()
	suspension.CreatedAt = now
	suspension.UpdatedAt = now

	stmt, args, err := toInsertStatement(tableSuspensions, suspension)
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	sqlResult, err := t.tx.ExecContext(ctx, stmt, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgInsertContext)
	}

	id, err := sqlResult.LastInsertId()
	if err != nil {
		return nil, errors.Wrap(err, msgRetrieveLastID)
	}
	return &id, nil
}

// Suspension returns suspension for given id
func (t *Transaction) Suspension(ctx context.Context, id int64) (*models.Suspension, error) {
	q := squirrel.Select("*").From(tableSuspensions).
		Where(squirrel.Eq{"id": id})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var suspensions []*models.Suspension
	err = t.tx.SelectContext(ctx, &suspensions, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	if len(suspensions) == 0 {
		return nil, nil
	}

	if len(suspensions) == 1 {
		return suspensions[0], nil
	}

	return nil, errors.Errorf("Found more than one suspension of id %d", id)
}

// Suspensions returns suspensions for given ids
func (t *Transaction) Suspensions(ctx context.Context, ids []int64) ([]*models.Suspension, error) {
	q := squirrel.Select("*").From(tableSuspensions).
		Where(squirrel.Eq{"id": ids})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var suspensions []*models.Suspension
	err = t.tx.SelectContext(ctx, &suspensions, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return suspensions, nil
}

// SuspensionPage returns reports up to a limit, resuming from offset
func (t *Transaction) SuspensionPage(ctx context.Context, filter *models.SuspensionFilter, sort *models.SuspensionSort, limit uint64, offset uint64) ([]*models.Suspension, *models.PageInfo, error) {
	var err error
	q := squirrel.Select("*").From(tableSuspensions).Limit(limit).Offset(offset)

	q, err = sortBy(q, sort)
	if err != nil {
		return nil, nil, err
	}

	q, err = filterBy(q, filter)
	if err != nil {
		return nil, nil, err
	}

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSQLConversion)
	}

	var suspensions []*models.Suspension
	err = t.tx.SelectContext(ctx, &suspensions, sql, args...)
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSelectContext)
	}

	q = squirrel.Select("count(*) as total").From(tableSuspensions)
	q, err = filterBy(q, filter)
	if err != nil {
		return nil, nil, err
	}
	sql, args, err = q.ToSql()
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSQLConversion)
	}

	var pageInfo []*models.PageInfo
	err = t.tx.SelectContext(ctx, &pageInfo, sql, args...)
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSelectContext)
	}

	if len(pageInfo) != 1 {
		return nil, nil, errors.New("Incorrect number of results returned for page info")
	}

	return suspensions, pageInfo[0], nil
}

// LatestSuspensionForUsers retrieves the last suspension for the user if it exists
func (t *Transaction) LatestSuspensionForUsers(ctx context.Context, userIDs []int64) ([]*models.Suspension, error) {
	q := squirrel.Select("main.*").
		From(fmt.Sprintf("%s main", tableSuspensions)).
		Where(squirrel.Eq(map[string]interface{}{
			"target_user_id": userIDs,
		})).
		Where("main.id = (select s.id from suspensions s where s.target_user_id = main.target_user_id order by s.id desc limit 1)")

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var suspensions []*models.Suspension
	err = t.tx.SelectContext(ctx, &suspensions, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return suspensions, nil
}
