package leviathan

import (
	"context"
	"fmt"

	"github.com/Masterminds/squirrel"
	"github.com/pkg/errors"

	"code.justin.tv/safety/datastore/models"
)

const (
	notIgnored = "`ignore` is FALSE"
)

// SuspensionGuides fetch all suspension guide records
func (t *Transaction) SuspensionGuides(ctx context.Context) ([]*models.SuspensionGuide, error) {
	q := squirrel.Select("*").
		From(tableSuspensionGuides).
		Where(notIgnored).
		OrderBy("detailed_reason_title")

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var result []*models.SuspensionGuide
	err = t.tx.SelectContext(ctx, &result, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return result, nil
}

// SuspensionGuideContents fetches the specific suspension guide content by code
func (t *Transaction) SuspensionGuideContents(ctx context.Context, contents []string) ([]*models.SuspensionGuideContent, error) {
	q := squirrel.Select("*").
		From(tableSuspensionGuideContents).
		Where(squirrel.Eq{"content": contents})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var results []*models.SuspensionGuideContent
	err = t.tx.SelectContext(ctx, &results, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return results, nil
}

// SuspensionGuideContentsAll fetch all suspension guide content records
func (t *Transaction) SuspensionGuideContentsAll(ctx context.Context) ([]*models.SuspensionGuideContent, error) {
	q := squirrel.Select("*").From(tableSuspensionGuideContents).OrderBy("content_description")

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var result []*models.SuspensionGuideContent
	err = t.tx.SelectContext(ctx, &result, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return result, nil
}

// SuspensionGuideReasons fetches the specific suspension guide reason by code
func (t *Transaction) SuspensionGuideReasons(ctx context.Context, reasons []string) ([]*models.SuspensionGuideReason, error) {
	q := squirrel.Select("*").
		From(tableSuspensionGuideReasons).
		Where(squirrel.Eq{"reason": reasons})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var results []*models.SuspensionGuideReason
	err = t.tx.SelectContext(ctx, &results, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return results, nil
}

// SuspensionGuideDetailedReasonsByCode fetches the specific suspension guide detailed reason by detailed_reason field
func (t *Transaction) SuspensionGuideDetailedReasonsByCode(ctx context.Context, detailedReasons []string) ([]*models.SuspensionGuideDetailedReason, error) {
	q := squirrel.Select("*").
		From(tableSuspensionGuideDetailedReasons).
		Where(squirrel.Eq{"detailed_reason": detailedReasons})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var results []*models.SuspensionGuideDetailedReason
	err = t.tx.SelectContext(ctx, &results, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return results, nil
}

// SuspensionGuideReasonsAll fetch all suspension guide reason records
func (t *Transaction) SuspensionGuideReasonsAll(ctx context.Context) ([]*models.SuspensionGuideReason, error) {
	q := squirrel.Select("*").From(tableSuspensionGuideReasons)

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var result []*models.SuspensionGuideReason
	err = t.tx.SelectContext(ctx, &result, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return result, nil
}

// SuspensionGuideDetailedReasonsAll fetch all suspension guide detailed reason records
func (t *Transaction) SuspensionGuideDetailedReasonsAll(ctx context.Context) ([]*models.SuspensionGuideDetailedReason, error) {
	q := squirrel.Select("*").From(tableSuspensionGuideDetailedReasons)

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var result []*models.SuspensionGuideDetailedReason
	err = t.tx.SelectContext(ctx, &result, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return result, nil
}

// SuspensionGuideReasonsByContent fetch all suspension guide reason records for the given content
func (t *Transaction) SuspensionGuideReasonsByContent(ctx context.Context, content string) ([]*models.SuspensionGuideReason, error) {
	q := squirrel.Select(fmt.Sprintf("distinct %s.*", tableSuspensionGuideReasons)).
		From(fmt.Sprintf("%s, %s, %s", tableSuspensionGuideReasons, tableSuspensionGuides, tableSuspensionGuideContents)).
		Where(fmt.Sprintf("%s.id = %s.reason_id", tableSuspensionGuideReasons, tableSuspensionGuides)).
		Where(notIgnored).
		Where(fmt.Sprintf("%s.content_id = %s.id", tableSuspensionGuides, tableSuspensionGuideContents)).
		Where(fmt.Sprintf("%s.content = ?", tableSuspensionGuideContents), content).
		OrderBy("reason_description")

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var result []*models.SuspensionGuideReason
	err = t.tx.SelectContext(ctx, &result, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return result, nil
}

// SuspensionGuideDetailedEntries fetch all suspension guide detailed entries for a given content and reason
func (t *Transaction) SuspensionGuideDetailedEntries(ctx context.Context, content, reason string) ([]*models.SuspensionGuideDetailedEntry, error) {
	// SQLX doesn't know how to populate data when there's multiple columns with same name.
	// Need to explicitly list the column names besides `suspension_guides`
	q := squirrel.Select(fmt.Sprintf("%s.*, %s.content, %s.reason, %s.detailed_reason, %s.points, %s.user_notification, %s.warnable",
		tableSuspensionGuides, tableSuspensionGuideContents, tableSuspensionGuideReasons, tableSuspensionGuideDetailedReasons, tableSuspensionGuideDetailedReasons, tableSuspensionGuideDetailedReasons, tableSuspensionGuideDetailedReasons)).
		From(fmt.Sprintf("%s, %s, %s, %s", tableSuspensionGuides, tableSuspensionGuideContents, tableSuspensionGuideReasons, tableSuspensionGuideDetailedReasons)).
		Where(notIgnored).
		Where(fmt.Sprintf("%s.id = %s.content_id", tableSuspensionGuideContents, tableSuspensionGuides)).
		Where(fmt.Sprintf("%s.id = %s.reason_id", tableSuspensionGuideReasons, tableSuspensionGuides)).
		Where(fmt.Sprintf("%s.id = %s.detailed_reason_id", tableSuspensionGuideDetailedReasons, tableSuspensionGuides)).
		Where(fmt.Sprintf("%s.content = ?", tableSuspensionGuideContents), content).
		Where(fmt.Sprintf("%s.reason = ?", tableSuspensionGuideReasons), reason).
		OrderBy("detailed_reason_title")

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var result []*models.SuspensionGuideDetailedEntry
	err = t.tx.SelectContext(ctx, &result, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return result, nil
}

// SuspensionGuideQuickSelects fetch all suspension guide entries that should be listed in quick select
func (t *Transaction) SuspensionGuideQuickSelects(ctx context.Context) ([]*models.SuspensionGuideDetailedEntry, error) {
	// SQLX doesn't know how to populate data when there's multiple columns with same name.
	// Need to explicitly list the column names besides `suspension_guides`
	q := squirrel.Select(fmt.Sprintf("%s.*, %s.content, %s.reason, %s.detailed_reason, %s.points, %s.user_notification, %s.warnable", tableSuspensionGuides, tableSuspensionGuideContents,
		tableSuspensionGuideReasons, tableSuspensionGuideDetailedReasons, tableSuspensionGuideDetailedReasons, tableSuspensionGuideDetailedReasons, tableSuspensionGuideDetailedReasons)).
		From(fmt.Sprintf("%s, %s, %s, %s", tableSuspensionGuides, tableSuspensionGuideContents, tableSuspensionGuideReasons, tableSuspensionGuideDetailedReasons)).
		Where(notIgnored).
		Where("list_in_quick_select is TRUE").
		Where(fmt.Sprintf("%s.id = %s.content_id", tableSuspensionGuideContents, tableSuspensionGuides)).
		Where(fmt.Sprintf("%s.id = %s.reason_id", tableSuspensionGuideReasons, tableSuspensionGuides)).
		Where(fmt.Sprintf("%s.id = %s.detailed_reason_id", tableSuspensionGuideDetailedReasons, tableSuspensionGuides)).
		OrderBy("detailed_reason_title")

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var result []*models.SuspensionGuideDetailedEntry
	err = t.tx.SelectContext(ctx, &result, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return result, nil
}
