package leviathan

import (
	"context"
	"time"

	"code.justin.tv/safety/datastore/models"
	"github.com/Masterminds/squirrel"
	"github.com/pkg/errors"
)

// CreateUserContentAudit creates a new user generated content audit record in database
func (t *Transaction) CreateUserContentAudit(ctx context.Context, audit models.UserContentAudit) (*int64, error) {
	now := time.Now()
	audit.CreatedAt = now
	audit.UpdatedAt = now

	sql, args, err := toInsertStatement(tableUserContentAudits, audit)
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	sqlResult, err := t.tx.ExecContext(ctx, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgInsertContext)
	}

	id, err := sqlResult.LastInsertId()
	if err != nil {
		return nil, errors.Wrap(err, msgRetrieveLastID)
	}

	return &id, nil
}

// CreateUserContentAudits creates new user content audit records in database
func (t *Transaction) CreateUserContentAudits(ctx context.Context, audits []*models.UserContentAudit) error {
	now := time.Now()
	if len(audits) == 0 {
		return nil
	}

	// Get the columns from the first row, all of them will have the same
	columns, _ := toColumnsAndValues(audits[0])
	sb := squirrel.Insert(tableUserContentAudits).Columns(columns...)
	for _, audit := range audits {
		audit.CreatedAt = now
		audit.UpdatedAt = now

		_, values := toColumnsAndValues(audit)
		sb = sb.Values(values...)
	}

	sql, args, err := sb.ToSql()
	if err != nil {
		return errors.Wrap(err, msgSQLConversion)
	}

	_, err = t.tx.ExecContext(ctx, sql, args...)
	if err != nil {
		return errors.Wrap(err, msgInsertContext)
	}

	return nil
}

// UserContentAudits returns the audit entries for a given ids
func (t *Transaction) UserContentAudits(ctx context.Context, ids []int64) ([]*models.UserContentAudit, error) {
	q := squirrel.Select("*").From(tableUserContentAudits).
		Where(squirrel.Eq{"id": ids})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var audits []*models.UserContentAudit
	err = t.tx.SelectContext(ctx, &audits, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	if len(audits) == 0 {
		return nil, nil
	}

	return audits, nil
}

// UserContentAuditByUser returns a paginated ugc audits for a given user
func (t *Transaction) UserContentAuditByUser(ctx context.Context, userID string, limit, offset uint64) ([]*models.UserContentAudit, *models.PageInfo, error) {
	q := squirrel.Select("*").From(tableUserContentAudits).
		Where(squirrel.Eq{"user_id": userID}).
		Limit(limit).
		Offset(offset).
		OrderBy("created_at desc")

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSQLConversion)
	}

	var audits []*models.UserContentAudit
	err = t.tx.SelectContext(ctx, &audits, sql, args...)
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSelectContext)
	}

	q = squirrel.Select("count(*) as total").
		Where(squirrel.Eq{"user_id": userID}).
		From(tableUserContentAudits)

	sql, args, err = q.ToSql()
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSQLConversion)
	}

	var pageInfo []*models.PageInfo
	err = t.tx.SelectContext(ctx, &pageInfo, sql, args...)
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSelectContext)
	}

	if len(pageInfo) != 1 {
		return nil, nil, errors.New("Incorrect number of results returned for page info")
	}

	return audits, pageInfo[0], nil
}
