package leviathan

import (
	"context"
	"time"

	"github.com/Masterminds/squirrel"

	"code.justin.tv/safety/datastore/models"
	"github.com/pkg/errors"
)

// CreateWarning creates a new warning record in database
func (t *Transaction) CreateWarning(ctx context.Context, warning models.Warning) (*int64, error) {
	now := time.Now()
	warning.CreatedAt = now
	warning.UpdatedAt = now

	stmt, args, err := toInsertStatement(tableWarnings, warning)
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	sqlResult, err := t.tx.ExecContext(ctx, stmt, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgInsertContext)
	}

	id, err := sqlResult.LastInsertId()
	if err != nil {
		return nil, errors.Wrap(err, msgRetrieveLastID)
	}
	return &id, nil
}

// Warning returns the warning for a given id
func (t *Transaction) Warning(ctx context.Context, id int64) (*models.Warning, error) {
	warnings, err := t.Warnings(ctx, []int64{id})
	if err != nil {
		return nil, err
	}

	if len(warnings) == 0 {
		return nil, nil
	}

	if len(warnings) == 1 {
		return warnings[0], nil
	}

	return nil, errors.Errorf("Found more than one warning of id %d", id)
}

// UpdateWarning updates an existing Warning in place
func (t *Transaction) UpdateWarning(ctx context.Context, warning models.Warning) error {
	warning.UpdatedAt = time.Now()

	sql, args, err := toUpdateStatement(tableWarnings, warning)
	if err != nil {
		return errors.Wrap(err, msgSQLConversion)
	}

	_, err = t.tx.ExecContext(ctx, sql, args...)
	if err != nil {
		return errors.Wrap(err, msgUpdateContext)
	}

	return nil
}

// Warnings returns warnings for given ids
func (t *Transaction) Warnings(ctx context.Context, ids []int64) ([]*models.Warning, error) {
	q := squirrel.Select("*").From(tableWarnings).
		Where(squirrel.Eq{"id": ids})

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, errors.Wrap(err, msgSQLConversion)
	}

	var warnings []*models.Warning
	err = t.tx.SelectContext(ctx, &warnings, sql, args...)
	if err != nil {
		return nil, errors.Wrap(err, msgSelectContext)
	}

	return warnings, nil
}

// WarningPage returns warnings up to a limit, resuming from offset
func (t *Transaction) WarningPage(ctx context.Context, filter *models.WarningFilter, sort *models.WarningSort, limit uint64, offset uint64) ([]*models.Warning, *models.PageInfo, error) {
	var err error
	q := squirrel.Select("*").From(tableWarnings).Limit(limit).Offset(offset)

	q, err = sortBy(q, sort)
	if err != nil {
		return nil, nil, err
	}

	q, err = filterBy(q, filter)
	if err != nil {
		return nil, nil, err
	}

	sql, args, err := q.ToSql()
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSQLConversion)
	}

	var warnings []*models.Warning
	err = t.tx.SelectContext(ctx, &warnings, sql, args...)
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSelectContext)
	}

	q = squirrel.Select("count(*) as total").From(tableWarnings)
	q, err = filterBy(q, filter)
	if err != nil {
		return nil, nil, err
	}
	sql, args, err = q.ToSql()
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSQLConversion)
	}

	var pageInfo []*models.PageInfo
	err = t.tx.SelectContext(ctx, &pageInfo, sql, args...)
	if err != nil {
		return nil, nil, errors.Wrap(err, msgSelectContext)
	}

	if len(pageInfo) != 1 {
		return nil, nil, errors.New("Incorrect number of results returned for page info")
	}

	return warnings, pageInfo[0], nil
}
