package models

import (
	"encoding/json"
	"fmt"

	"github.com/pkg/errors"
)

var (
	// ErrUnexpectedNumberOfFields is returned when unexpected number of fields are found
	ErrUnexpectedNumberOfFields = fmt.Errorf("Unexpected number of fields found")

	reportAuditSkippedFields = map[string]struct{}{
		"CreatedAt": struct{}{},
		"UpdatedAt": struct{}{},
	}
)

// AuditDiff generates diff used to audit report
func AuditDiff(oldReport, newReport *Report) (string, error) {
	oldValues, err := oldReport.toValueMap()
	if err != nil {
		return "", err
	}
	newValues, err := newReport.toValueMap()
	if err != nil {
		return "", err
	}

	diffMap := map[string][]interface{}{}

	// two maps should always have same set of keys, sanity check
	if len(oldValues) != len(newValues) {
		return "", errors.WithStack(ErrUnexpectedNumberOfFields)
	}

	for key, value := range oldValues {
		// Skip the field if we don't care about it
		if _, ok := reportAuditSkippedFields[key]; ok {
			continue
		}

		vn := newValues[key]
		if value != vn {
			diffMap[key] = []interface{}{value, vn}
		}
	}

	d, err := json.Marshal(diffMap)
	if err != nil {
		return "", err
	}

	return string(d), err
}

func (r *Report) toValueMap() (map[string]interface{}, error) {
	b, err := json.Marshal(r)
	if err != nil {
		return nil, err
	}

	result := map[string]interface{}{}
	err = json.Unmarshal(b, &result)
	if err != nil {
		return nil, err
	}
	return result, nil
}
