package models

import (
	"time"

	"code.justin.tv/safety/datastore/models/shift"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

// PageInfo stores information about a page retrieved from the database
// to determine if we are on the first or last page, and how many Total
// items there are
type PageInfo struct {
	FirstID int64 `db:"first_id"` // DEPRECATED: this does not work for sorted result
	LastID  int64 `db:"last_id"`  // DEPRECATED: this does not work for sorted result
	Total   int64 `db:"total"`
}

// ReplacementInfo stores information about a replacement
type ReplacementInfo struct {
	Removed int64
	Created int64
}

// Report is a row of reports
type Report struct {
	ID                    int64     `db:"id"`
	Status                *string   `db:"status"`
	AssignedTo            *int64    `db:"assigned_to"`
	AssignedToQueue       int64     `db:"assigned_to_queue"`
	FromUserUsername      *string   `db:"from_user_username"`
	FromUserID            *int      `db:"from_user_id"`
	FromUserType          *string   `db:"from_user_type"`
	FromUserPartner       *bool     `db:"from_user_partner"`
	TargetUserUsername    *string   `db:"target_user_username"`
	TargetUserID          *int      `db:"target_user_id"`
	TargetUserType        *string   `db:"target_user_type"`
	TargetUserPartner     *bool     `db:"target_user_partner"`
	Content               *string   `db:"content"`
	ContentID             *string   `db:"content_id"`
	Reason                *string   `db:"reason"`
	Description           *string   `db:"description"`
	FromUserDisplayName   *string   `db:"from_user_display_name"`
	TargetUserDisplayName *string   `db:"target_user_display_name"`
	Origin                *string   `db:"origin"`
	Extra1                *string   `db:"extra1"`
	Details               *string   `db:"details"`
	TargetUserGame        *string   `db:"target_user_game"`
	Language              *string   `db:"language"`
	CreatedAt             time.Time `db:"created_at"`
	UpdatedAt             time.Time `db:"updated_at"`
}

// ReportComment is a row of report_comments
type ReportComment struct {
	ID        int64     `db:"id"`
	ReportID  int64     `db:"report_id"`
	CreatedBy int64     `db:"created_by"`
	Comment   string    `db:"comment"`
	CreatedAt time.Time `db:"created_at"`
	UpdatedAt time.Time `db:"updated_at"`
}

// Admin is a row of users
type Admin struct {
	ID                      int64     `db:"id"`
	Enabled                 bool      `db:"enabled"`
	ReportSoundNotification bool      `db:"report_sound_notification"`
	ReportMode              string    `db:"report_mode"`
	Style                   string    `db:"style"`
	Email                   *string   `db:"email"`
	LdapID                  string    `db:"ldap_id"`
	CreatedAt               time.Time `db:"created_at"`
	UpdatedAt               time.Time `db:"updated_at"`
}

// ReportAudit a row of report_audits
type ReportAudit struct {
	ID            int64     `db:"id"`
	ReportID      *int64    `db:"report_id"`
	Action        *string   `db:"action"`
	ActionBy      *int64    `db:"action_by"`
	AutoResolveID *int64    `db:"auto_resolve_id"`
	Diff          *string   `db:"diff"`
	HoldID        *int64    `db:"hold_id"`
	CreatedAt     time.Time `db:"created_at"`
	UpdatedAt     time.Time `db:"updated_at"`
}

// ReportHold a row of report_holds
type ReportHold struct {
	ID         int64      `db:"id"`
	ReportID   int64      `db:"report_id"`
	CreatedBy  int64      `db:"created_by"`
	HoldUntil  time.Time  `db:"hold_until"`
	DisabledAt *time.Time `db:"disabled_at"`
	CreatedAt  time.Time  `db:"created_at"`
	UpdatedAt  time.Time  `db:"updated_at"`
}

// AutoResolve is a row of auto_resolves
type AutoResolve struct {
	ID           int64      `db:"id"`
	TargetUserID *int       `db:"target_user_id"`
	FromUserID   *int       `db:"from_user_id"`
	Content      *string    `db:"content"`
	Reason       *string    `db:"reason"`
	ContentID    *string    `db:"content_id"`
	DisabledAt   *time.Time `db:"disabled_at"`
	CreatedBy    *int64     `db:"created_by"`
	ResolveUtil  *time.Time `db:"resolve_until"`
	Count        int64      `db:"count"`
	CreatedAt    time.Time  `db:"created_at"`
	UpdatedAt    time.Time  `db:"updated_at"`
}

// AutoResolveUserOperation is the type of operation of the auto resolve user
// These would be operations like IS user and IS NOT the user
type AutoResolveUserOperation string

// AutoResolveUser describes an user that's being auto resolved
type AutoResolveUser struct {
	UserID        int
	OperationType AutoResolveUserOperation
}

// Suspension is a row of suspensions
type Suspension struct {
	ID                    int64     `db:"id"`
	FromUserID            *string   `db:"from_user_id"`
	FromUserType          *string   `db:"from_user_type"`
	TargetUserUsername    *string   `db:"target_user_username"`
	TargetUserID          *int64    `db:"target_user_id"`
	TargetUserType        *string   `db:"target_user_type"`
	TargetUserPartner     *bool     `db:"target_user_partner"`
	TargetUserDisplayName *string   `db:"target_user_display_name"`
	Content               *string   `db:"content"`
	Reason                *string   `db:"reason"`
	DetailedReason        *string   `db:"detailed_reason"`
	Description           *string   `db:"description"`
	Duration              *int64    `db:"duration"`
	IPBan                 *int64    `db:"ip_ban"`
	ClearedChannelImages  *bool     `db:"cleared_channel_images"`
	Origin                *string   `db:"origin"`
	TargetUserGame        *string   `db:"target_user_game"`
	CurrentLiveVod        *int      `db:"current_live_vod"`
	CreatedAt             time.Time `db:"created_at"`
	UpdatedAt             time.Time `db:"updated_at"`
}

// Warning is a row of suspensions
type Warning struct {
	ID                    int64     `db:"id"`
	FromUserID            string    `db:"from_user_id"`
	FromUserType          *string   `db:"from_user_type"`
	TargetUserUsername    string    `db:"target_user_username"`
	TargetUserDisplayName string    `db:"target_user_display_name"`
	TargetUserID          string    `db:"target_user_user_id"`
	TargetUserType        *string   `db:"target_user_type"`
	TargetUserPartner     *bool     `db:"target_user_partner"`
	Content               string    `db:"content"`
	Reason                string    `db:"reason"`
	DetailedReason        string    `db:"detailed_reason"`
	Description           string    `db:"description"`
	Origin                *string   `db:"origin"`
	ClearedChannelImages  bool      `db:"cleared_channel_images"`
	TargetUserGame        *string   `db:"target_user_game"`
	Seen                  bool      `db:"seen"`
	CreatedAt             time.Time `db:"created_at"`
	UpdatedAt             time.Time `db:"updated_at"`
}

// Enforcement is the db representation of a generic enforcement action
type Enforcement struct {
	ID                int64     `db:"id"`
	Reason            string    `db:"reason"`
	DetailedReason    string    `db:"detailed_reason"`
	FromUserID        string    `db:"from_user_id"`
	TargetUserID      string    `db:"target_user_user_id"`
	Content           string    `db:"content"`
	Description       string    `db:"description"`
	EnforcementType   string    `db:"enforcement_type"`
	EnforcementID     string    `db:"enforcement_id"`
	TargetUserType    *string   `db:"target_user_type"`
	TargetUserPartner *bool     `db:"target_user_partner"`
	TargetUserGame    *string   `db:"target_user_game"`
	CreatedAt         time.Time `db:"created_at"`
	UpdatedAt         time.Time `db:"updated_at"`
}

// SuspensionCheck is a row of suspension_checks
type SuspensionCheck struct {
	ID                   int64     `db:"id"`
	SuspensionID         *int64    `db:"suspension_id"`
	SuspensionFromUserID *string   `db:"suspension_from_user_id"`
	Status               *string   `db:"status"`
	CheckedBy            *int64    `db:"checked_by"`
	CreatedAt            time.Time `db:"created_at"`
	UpdatedAt            time.Time `db:"updated_at"`
}

// SuspensionGuideContent is a row of suspension_guide_contents
type SuspensionGuideContent struct {
	ID                 int64     `db:"id"`
	Content            string    `db:"content"`
	ContentDescription string    `db:"content_description"`
	CreatedAt          time.Time `db:"created_at"`
	UpdatedAt          time.Time `db:"updated_at"`
}

// SuspensionGuideReason is a row of suspension_guide_reasons
type SuspensionGuideReason struct {
	ID                int64     `db:"id"`
	Reason            string    `db:"reason"`
	ReasonDescription string    `db:"reason_description"`
	CreatedAt         time.Time `db:"created_at"`
	UpdatedAt         time.Time `db:"updated_at"`
}

// SuspensionGuideDetailedReason is a row of suspension_guide_detailed_reasons
type SuspensionGuideDetailedReason struct {
	ID               int64     `db:"id"`
	DetailedReason   string    `db:"detailed_reason"`
	Points           int64     `db:"points"`
	UserNotification *string   `db:"user_notification"`
	Warnable         bool      `db:"warnable"`
	CreatedAt        time.Time `db:"created_at"`
	UpdatedAt        time.Time `db:"updated_at"`
}

// SuspensionGuide is a row of suspension_guides
type SuspensionGuide struct {
	ID                        int64     `db:"id"`
	Ignore                    bool      `db:"ignore"`
	IPBan                     bool      `db:"ip_ban"`
	Permanent                 bool      `db:"permanent"`
	ClearImages               bool      `db:"clear_images"`
	RemoveOffendingContent    bool      `db:"remove_offending_content"`
	Escalate                  bool      `db:"escalate"`
	ListInQuickSelect         bool      `db:"list_in_quick_select"`
	ContentID                 int64     `db:"content_id"`
	ReasonID                  int64     `db:"reason_id"`
	DetailedReasonID          int64     `db:"detailed_reason_id"`
	DetailedReasonTitle       string    `db:"detailed_reason_title"`
	DetailedReasonDescription string    `db:"detailed_reason_description"`
	StrikeDetails             string    `db:"strike_details"`
	StrikeFormNote            string    `db:"strike_form_note"`
	OffenseSendToUser         string    `db:"offense_send_to_user"`
	ViolatedTosSection        string    `db:"violated_tos_section"`
	CreatedAt                 time.Time `db:"created_at"`
	UpdatedAt                 time.Time `db:"updated_at"`
}

// SuspensionGuideDetailedEntry is a detailed entry of the suspension guide that
// combines suspension guide and detailed reason
type SuspensionGuideDetailedEntry struct {
	SuspensionGuide
	Content          string  `db:"content"`
	Reason           string  `db:"reason"`
	DetailedReason   string  `db:"detailed_reason"`
	Points           int64   `db:"points"`
	UserNotification *string `db:"user_notification"`
	Warnable         bool    `db:"warnable"`
}

// CursorPageInfo stores whether there are records before and after requested data
type CursorPageInfo struct {
	StartCursor *string
	EndCursor   *string
	HasNext     bool
	HasPrevious bool
}

// ChatEntry is a message or action taken in chat by a user
type ChatEntry struct {
	MessageID   string
	Timestamp   int64
	MessageType string
	ChannelID   string
	UserID      string

	Emotes        *string
	Badges        *string
	MessageText   *string
	SystemMessage *string

	NoticeType *string

	TimeoutDuration *int32
	TimeoutReason   *string

	IsDeleted *bool

	Cursor *string
	Key    *map[string]*dynamodb.AttributeValue
}

// TimeWindow describes a window of time
type TimeWindow struct {
	Start int64
	End   int64
}

// Shift describes a shift
type Shift struct {
	ID      string
	AdminID int64
	Type    shift.Type
	Weight  int64
}

// UserReportCount describes reports against a given user
type UserReportCount struct {
	UserID int64 `db:"user_id"`
	Count  int64 `db:"count"`
}

// BanException describes an entry in ban_exceptions table
type BanException struct {
	ID           int64     `db:"id"`
	TargetUserID int64     `db:"target_user_id"`
	CreatedBy    int64     `db:"created_by"`
	Reason       string    `db:"reason"`
	CreatedAt    time.Time `db:"created_at"`
	UpdatedAt    time.Time `db:"updated_at"`
}

// FileUpload describes a file upload entry
type FileUpload struct {
	ID        int64     `db:"id"`
	UserID    int64     `db:"user_id"`
	URL       string    `db:"url"`
	File      string    `db:"file"`
	Channel   *string   `db:"channel"`
	CreatedAt time.Time `db:"created_at"`
	UpdatedAt time.Time `db:"updated_at"`
}

// FileUploadAudit describes an audit entry for an uploaded file
type FileUploadAudit struct {
	ID        int64     `db:"id"`
	Action    string    `db:"action"`
	ActionBy  int64     `db:"action_by"`
	Content   string    `db:"content"`
	RemoteIP  string    `db:"remote_ip"`
	CreatedAt time.Time `db:"created_at"`
	UpdatedAt time.Time `db:"updated_at"`
}

// UserContentAudit describes an audit entry for actions against user generated content on twitch
type UserContentAudit struct {
	ID           int64     `db:"id"`
	Action       string    `db:"action"`
	ActionBy     int64     `db:"action_by"`
	ResourceID   string    `db:"resource_id"`
	ResourceType string    `db:"resource_type"`
	UserID       string    `db:"user_id"`
	CreatedAt    time.Time `db:"created_at"`
	UpdatedAt    time.Time `db:"updated_at"`
}

// EscalationPartner describes an entry on the Partner Escalation List
type EscalationPartner struct {
	UserID    string    `db:"user_id"`
	Override  *bool     `db:"override"`
	CreatedAt time.Time `db:"created_at"`
	UpdatedAt time.Time `db:"updated_at"`
}
