package reportrouter

import (
	"fmt"
	"time"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/go-redis/redis"
	"github.com/pkg/errors"
)

var (
	//ErrMissingConfiguration indiciates the configuration is nil
	ErrMissingConfiguration = fmt.Errorf("The configuration is nil")

	// ErrInvalidRedisHost indicates invalid redis host
	ErrInvalidRedisHost = fmt.Errorf("Missing redis host")

	// ErrMissingStatter indicates missing statsd Statter
	ErrMissingStatter = fmt.Errorf("Missing statter")
)

// Datastore defines APIs to access routing data
type Datastore struct {
	routerDB redis.Client
	statter  statsd.Statter
}

// Config defines connection details to Redis
type Config struct {
	RedisHost string
	Timeout   time.Duration
	Statter   statsd.Statter
}

// Validate validates the configuration options
func (c *Config) Validate() error {
	if c == nil {
		return ErrMissingConfiguration
	}
	if c.RedisHost == "" {
		return ErrInvalidRedisHost
	}
	if c.Statter == nil {
		return ErrMissingStatter
	}
	return nil
}

// New returns a new router datastore to access routing data
func New(c *Config) (*Datastore, error) {
	err := c.Validate()
	if err != nil {
		return nil, err
	}

	redis := redis.NewClient(&redis.Options{
		Addr:        c.RedisHost,
		ReadTimeout: c.Timeout,
	})

	err = redis.Ping().Err()
	if err != nil {
		return nil, errors.Wrap(err, "could not ping")
	}

	return &Datastore{
		routerDB: *redis,
		statter:  c.Statter,
	}, nil
}

func (r *Datastore) scanAll(match string) ([]string, error) {
	startTime := time.Now()
	var cursor uint64
	allKeys := []string{}
	for {
		var keys []string
		var err error
		keys, cursor, err = r.routerDB.Scan(cursor, match, 50).Result()
		if err != nil {
			return nil, errors.Wrap(err, "scanAll")
		}
		allKeys = append(allKeys, keys...)
		if cursor == 0 {
			break
		}
	}
	lag := time.Since(startTime)
	_ = r.statter.TimingDuration("scan", lag, 1)
	return allKeys, nil
}
