package reportrouter

import (
	"fmt"
	"math"
	"strconv"
	"time"

	"code.justin.tv/safety/datastore/models"
	"github.com/go-redis/redis"
	"github.com/pkg/errors"
)

const (
	routeExpiration = 4 * time.Hour
)

// CreateRoute creates or overwrites existing route for target user ID
func (r *Datastore) CreateRoute(shiftID string, targetUserID string) error {
	startTime := time.Now()
	err := r.routerDB.Set(routeKey(targetUserID), shiftID, routeExpiration).Err()
	_ = r.statter.TimingDuration("create_route", time.Since(startTime), 1)
	return errors.Wrap(err, "could not create route")
}

// RefreshRoute moves the expiration date on a route forward
func (r *Datastore) RefreshRoute(targetUserID string) error {
	startTime := time.Now()
	err := r.routerDB.Expire(routeKey(targetUserID), routeExpiration).Err()
	_ = r.statter.TimingDuration("refresh_route", time.Since(startTime), 1)
	return errors.Wrap(err, "could not refresh route")
}

// DeleteRoute deletes a route for target user ID
func (r *Datastore) DeleteRoute(targetUserID string) error {
	startTime := time.Now()
	err := r.routerDB.Del(routeKey(targetUserID)).Err()
	_ = r.statter.TimingDuration("delete_route", time.Since(startTime), 1)
	return errors.Wrap(err, "could not delete route")
}

// DeleteRoutes deletes routes belonging to a shift
func (r *Datastore) DeleteRoutes(shiftID string) error {
	startTime := time.Now()
	routes, err := r.scanAll("route:*")
	if err != nil {
		return errors.Wrap(err, "could not get routes")
	}
	for _, rKey := range routes {
		var routedShift string
		routedShift, err := r.routerDB.Get(rKey).Result()
		if err != nil {
			return errors.Wrap(err, "could not get route")
		}
		if routedShift == shiftID {
			err = r.routerDB.Del(rKey).Err()
			if err != nil {
				return errors.Wrapf(err, "could not delete route %s", rKey)
			}
		}
	}
	_ = r.statter.TimingDuration("delete_routes", time.Since(startTime), 1)
	return nil
}

// DeleteAllRoutes deletes all existing routes
func (r *Datastore) DeleteAllRoutes() error {
	startTime := time.Now()
	routeKeys, err := r.scanAll("route:*")
	if err != nil {
		return errors.Wrap(err, "could not get routes")
	}
	routeKeysCount := len(routeKeys)
	const step = 25
	for start := 0; start < routeKeysCount; start += step {
		end := start + step
		if end > routeKeysCount {
			end = routeKeysCount
		}
		err = r.routerDB.Del(routeKeys[start:end]...).Err()
		if err != nil {
			return errors.Wrap(err, "could not delete routes")
		}
	}
	_ = r.statter.TimingDuration("delete_all_routes", time.Since(startTime), 1)
	return nil
}

// RoutedReports returns report IDs routed to the shift
func (r *Datastore) RoutedReports(shiftID string) ([]int64, error) {
	startTime := time.Now()
	stringIDs, err := r.routerDB.LRange(reportsKey(shiftID), 0, math.MaxUint32).Result()
	if err != nil {
		return nil, errors.Wrap(err, "could not get reports")
	}
	var reportIDs []int64
	for _, sID := range stringIDs {
		var id int64
		id, err = strconv.ParseInt(sID, 10, 64)
		if err != nil {
			return nil, errors.Wrapf(err, "invalid report id %s", sID)
		}
		reportIDs = append(reportIDs, id)
	}
	_ = r.statter.TimingDuration("routed_reports", time.Since(startTime), 1)
	return reportIDs, nil
}

// Route returns the current route for target user ID
func (r *Datastore) Route(targetUserID string) (*models.Shift, error) {
	startTime := time.Now()
	sKey, err := r.routerDB.Get(routeKey(targetUserID)).Result()
	if err == redis.Nil {
		return nil, nil
	} else if err != nil {
		return nil, errors.Wrap(err, "could not get route")
	}
	_ = r.statter.TimingDuration("route", time.Since(startTime), 1)
	return r.Shift(sKey)
}

func routeKey(targetUserID string) string {
	return fmt.Sprintf("route:%s", targetUserID)
}

func reportsKey(shiftID string) string {
	return fmt.Sprintf("reports:%s", shiftID)
}
