package reportrouter

import (
	"fmt"
	"strconv"
	"time"

	"code.justin.tv/safety/datastore/models"
	"code.justin.tv/safety/datastore/models/shift"
	"github.com/pkg/errors"
)

var shiftFields = []string{"adminID", "type", "weight"}

// CreateShift creates or overwrites existing shift for (admin, type) key
func (r *Datastore) CreateShift(adminID int64, shiftType shift.Type, weight int64) (*models.Shift, error) {
	startTime := time.Now()
	shiftID := r.ShiftID(adminID, shiftType)
	shift := &models.Shift{
		ID:      shiftID,
		AdminID: adminID,
		Type:    shiftType,
		Weight:  weight,
	}
	err := r.routerDB.HMSet(shiftKey(shiftID), redisMap(*shift)).Err()
	if err != nil {
		return nil, errors.Wrap(err, "could not create shift")
	}
	_ = r.statter.TimingDuration("create_shift", time.Since(startTime), 1)
	return shift, nil
}

// DeleteShift deletes a shift
func (r *Datastore) DeleteShift(shiftID string) error {
	startTime := time.Now()
	err := r.routerDB.Del(shiftKey(shiftID)).Err()
	_ = r.statter.TimingDuration("delete_shift", time.Since(startTime), 1)
	if err != nil {
		return errors.Wrap(err, "could not delete shift")
	}
	return r.RemoveAllReportsFromShift(shiftID)
}

// AddReportToShift adds routing for a specific report
func (r *Datastore) AddReportToShift(shiftID string, reportID int64) error {
	startTime := time.Now()
	err := r.routerDB.LPush(reportsKey(shiftID), strconv.FormatInt(reportID, 10)).Err()
	_ = r.statter.TimingDuration("add_report_to_shift", time.Since(startTime), 1)
	return errors.Wrap(err, "could not add report to shift")
}

// RemoveReportFromShift deletes routing from a report
func (r *Datastore) RemoveReportFromShift(shiftID string, reportID int64) error {
	startTime := time.Now()
	err := r.routerDB.LRem(reportsKey(shiftID), 0, strconv.FormatInt(reportID, 10)).Err()
	_ = r.statter.TimingDuration("remove_report_from_shift", time.Since(startTime), 1)
	return errors.Wrap(err, "could not remove report from shift")
}

// RemoveAllReportsFromShift deletes all routing from a report
func (r *Datastore) RemoveAllReportsFromShift(shiftID string) error {
	startTime := time.Now()
	err := r.routerDB.Del(reportsKey(shiftID)).Err()
	_ = r.statter.TimingDuration("remove_all_reports_from_shift", time.Since(startTime), 1)
	return errors.Wrap(err, "could not remove all reports from shift")
}

// Shift returns a single shift
func (r *Datastore) Shift(shiftID string) (*models.Shift, error) {
	return r.shift(shiftKey(shiftID))
}

func (r *Datastore) shift(key string) (*models.Shift, error) {
	startTime := time.Now()
	values, err := r.routerDB.HMGet(key, shiftFields...).Result()
	if err != nil {
		return nil, errors.Wrap(err, "could not get shift")
	} else if values == nil || allNils(values) {
		return nil, nil
	}
	_ = r.statter.TimingDuration("shift", time.Since(startTime), 1)
	adminID, err := strconv.ParseInt((values[0].(string)), 10, 64)
	if err != nil {
		return nil, errors.Wrap(err, "invalid adminID")
	}
	st, ok := values[1].(string)
	if !ok {
		return nil, errors.Errorf("invalid shiftType: %T", values[1])
	}
	shiftType := shift.Type(st)
	weight, err := strconv.ParseInt((values[2].(string)), 10, 64)
	if err != nil {
		return nil, errors.Wrap(err, "invalid weight on shift")
	}
	return &models.Shift{
		ID:      r.ShiftID(adminID, shiftType),
		AdminID: adminID,
		Type:    shiftType,
		Weight:  weight,
	}, nil
}

// Shifts returns all shifts
func (r *Datastore) Shifts() ([]models.Shift, error) {
	return r.shiftScan("shift:*")
}

// ShiftsByAdmin returns all shifts for an admin
func (r *Datastore) ShiftsByAdmin(adminID int64) ([]models.Shift, error) {
	return r.shiftScan(fmt.Sprintf("shift:%d:*", adminID))
}

func (r *Datastore) shiftScan(match string) ([]models.Shift, error) {
	startTime := time.Now()
	shiftKeys, err := r.scanAll(match)
	if err != nil {
		return nil, errors.Wrap(err, "could not get shifts")
	}
	var shifts []models.Shift
	for _, key := range shiftKeys {
		var shift *models.Shift
		shift, err = r.shift(key)
		if err != nil {
			return nil, errors.Wrap(err, "could not get shift")
		} else if shift != nil {
			// ignore non-existent keys caused by possible race with deleting
			// not important for our current use cases
			shifts = append(shifts, *shift)
		}
	}
	_ = r.statter.TimingDuration("shifts", time.Since(startTime), 1)
	return shifts, nil
}

// ShiftID constructs the Shift's ID
func (r *Datastore) ShiftID(adminID int64, shiftType shift.Type) string {
	return fmt.Sprintf("%d:%s", adminID, shiftType)
}

// IsValidType returns whether the type is a valid shift type
func (r *Datastore) IsValidType(shiftType shift.Type) bool {
	_, ok := shift.ValidTypes[shiftType]
	return ok
}

func shiftKey(shiftID string) string {
	return fmt.Sprintf("shift:%s", shiftID)
}

// convenience function to turn a shift into map for redis.HMSet
func redisMap(shift models.Shift) map[string]interface{} {
	return map[string]interface{}{
		"adminID": shift.AdminID,
		"type":    string(shift.Type),
		"weight":  shift.Weight,
	}
}

func allNils(array []interface{}) bool {
	for _, val := range array {
		if val != nil {
			return false
		}
	}
	return true
}
