package middleware

import (
	"bufio"
	"fmt"
	"net"
	"net/http"
	"time"

	"code.justin.tv/chat/golibs/logx"
)

// LogRequest returns a middleware that logs the path, duration and
// result of an http request.
//
// We do not use a generic solution like github.com/unrolled/logger because
// we want to use logx.
func LogRequest(inner http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		start := time.Now()

		crw := newCustomResponseWriter(w)
		inner.ServeHTTP(crw, r)
		logx.Info(r.Context(), "Handled request", logx.Fields{
			"duration": time.Since(start),
			"method":   r.Method,
			"size":     crw.size,
			"status":   crw.status,
			"uri":      r.RequestURI,
		})
	})
}

// customResponseWriter wraps a standard responseWriter and stores the
// header value and size of the response so that the logger can log those
// out
// customResponseWriter taken from https://github.com/unrolled/logger/blob/master/logger.go
// which is MIT licenced
type customResponseWriter struct {
	http.ResponseWriter
	status int
	size   int
}

func (c *customResponseWriter) WriteHeader(status int) {
	c.status = status
	c.ResponseWriter.WriteHeader(status)
}

func (c *customResponseWriter) Write(b []byte) (int, error) {
	size, err := c.ResponseWriter.Write(b)
	c.size += size
	return size, err
}

func (c *customResponseWriter) Flush() {
	if f, ok := c.ResponseWriter.(http.Flusher); ok {
		f.Flush()
	}
}

func (c *customResponseWriter) Hijack() (net.Conn, *bufio.ReadWriter, error) {
	if hj, ok := c.ResponseWriter.(http.Hijacker); ok {
		return hj.Hijack()
	}
	return nil, nil, fmt.Errorf("ResponseWriter does not implement the Hijacker interface")
}

func newCustomResponseWriter(w http.ResponseWriter) *customResponseWriter {
	// When WriteHeader is not called, it's safe to assume the status will be 200.
	return &customResponseWriter{
		ResponseWriter: w,
		status:         200,
	}
}
